/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.identity.idm.impl.api.session.managers;

import org.jboss.identity.idm.api.RelationshipManager;
import org.jboss.identity.idm.api.IdentitySession;
import org.jboss.identity.idm.api.Group;
import org.jboss.identity.idm.api.User;
import org.jboss.identity.idm.api.IdentityType;
import org.jboss.identity.idm.api.RelationshipManagerFeaturesDescription;
import org.jboss.identity.idm.api.IdentitySearchCriteria;
import org.jboss.identity.idm.api.IdentitySearchCriteriumType;
import org.jboss.identity.idm.exception.IdentityException;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationshipType;
import org.jboss.identity.idm.spi.model.IdentityObjectRelationship;
import org.jboss.identity.idm.spi.model.IdentityObject;
import org.jboss.identity.idm.spi.model.IdentityObjectType;
import org.jboss.identity.idm.spi.store.IdentityObjectSearchCriteriaType;
import org.jboss.identity.idm.impl.NotYetImplementedException;
import org.jboss.identity.idm.impl.api.session.managers.AbstractManager;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.util.HashSet;
import java.util.Arrays;

/**
 * @author <a href="mailto:boleslaw.dawidowicz at redhat.com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class RelationshipManagerImpl extends AbstractManager implements RelationshipManager
{
   RelationshipManagerFeaturesDescription featuresDescription;

   public static final IdentityObjectRelationshipType MEMBER = new IdentityObjectRelationshipType()
   {
      public String getName()
      {
         return "JBOSS_IDENTITY_MEMBERSHIP";
      }
   };

   public RelationshipManagerImpl(IdentitySession session)
   {
      super(session);

      featuresDescription = new RelationshipManagerFeaturesDescription()
      {
         public boolean isIdentityAssociationSupported(String fromGroupType)
         {

            IdentityObjectType identityOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();
            IdentityObjectType groupIdentityOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(fromGroupType);

            try
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRelationshipTypeSupported(groupIdentityOT, identityOT, MEMBER);
            }
            catch (IdentityException e)
            {
               return false;
            }

         }

         public boolean isGroupAssociationSupported(String fromGroupType, String toGroupType)
         {
            IdentityObjectType toGroupOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(toGroupType);
            IdentityObjectType fromGroupOT = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(fromGroupType);

            try
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isRelationshipTypeSupported(fromGroupOT, toGroupOT, MEMBER);
            }
            catch (IdentityException e)
            {
               return false;
            }
         }

         public boolean isIdentitiesSearchCriteriumTypeSupported(IdentitySearchCriteriumType constraintType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType();

            IdentityObjectSearchCriteriaType constraint = IdentityObjectSearchCriteriaType.valueOf(constraintType.name());


            if (constraint != null)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isSearchCriteriaTypeSupported(objectType, constraint);
            }
            
            return false;
         }

         public boolean isGroupsSearchCriteriumTypeSupported(String groupType, IdentitySearchCriteriumType constraintType)
         {
            IdentityObjectType objectType = getSessionContext().getIdentityObjectTypeMapper().getIdentityObjectType(groupType);

            IdentityObjectSearchCriteriaType constraint = IdentityObjectSearchCriteriaType.valueOf(constraintType.name());

            if (constraint != null)
            {
               return getSessionContext().getIdentityStoreRepository().getSupportedFeatures().
                  isSearchCriteriaTypeSupported(objectType, constraint);
            }

            return false;
         }
      };
   }

   public RelationshipManagerFeaturesDescription getFeaturesDescription()
   {
      return featuresDescription;
   }

   public void associateGroups(Collection<Group> parents, Collection<Group> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Group> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Group member = membersIterator.next();

            associateGroups(parent, member);
         }
      }
   }

   public void associateGroups(Group parent, Collection<Group> members) throws IdentityException
   {
      checkNotNullArgument(parent, "parent");
      checkNotNullArgument(members, "members");

      associateGroups(Arrays.asList(parent), members);
   }

   public void associateGroupsByIds(Collection<String> parentIds, Collection<String> memberIds) throws IdentityException
   {
      checkNotNullArgument(parentIds, "Parents Ids");
      checkNotNullArgument(memberIds, "Members Ids");

      for (Iterator<String> parentsIterator = parentIds.iterator(); parentsIterator.hasNext();)
      {
         String parentId = parentsIterator.next();

         for (Iterator<String> membersIterator = memberIds.iterator(); membersIterator.hasNext();)
         {
            String memberId = membersIterator.next();

            associateGroupsByIds(parentId, memberId);
         }
      }
   }

   public void associateGroupsByIds(String parentId, Collection<String> memberIds) throws IdentityException
   {
      checkNotNullArgument(parentId, "Parent Id");

      associateGroupsByIds(Arrays.asList(parentId), memberIds);
   }

   public void associateGroups(Group parent, Group member) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent group");
      checkNotNullArgument(member, "Member group");

      getRepository().createRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null, true);
   }
                           

   public void associateGroupsByIds(String parentId, String memberId) throws IdentityException
   {
      checkNotNullArgument(parentId, "Parent Id");
      checkNotNullArgument(memberId, "Member Id");

      getRepository().createRelationship(getInvocationContext(), createIdentityObjectForGroupId(parentId), createIdentityObjectForGroupId(memberId), MEMBER, null, true);
   }

   public void associateUsers(Collection<Group> parents, Collection<User> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<User> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            User member = membersIterator.next();

            associateUser(parent, member);
         }
      }
   }

   public void associateUser(Group parent, Collection<User> members) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent group");

      associateUsers(Arrays.asList(parent), members);

   }

   public void associateUsersByIds(Collection<String> parents, Collection<String> members) throws IdentityException
   {

      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<String> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         String parent = parentsIterator.next();

         for (Iterator<String> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            String member = membersIterator.next();

            associateUserByIds(parent, member);
         }
      }

   }

   public void associateUsersByIds(String parentId, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parentId, "Parent Id");

      associateUsersByIds(Arrays.asList(parentId), members);

   }

   public void associateUser(Group parent, User member) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent group");
      checkNotNullArgument(member, "Member user");

      getRepository().createRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null, true);
   }


   public void associateUserByIds(String parentId, String memberId) throws IdentityException
   {
      checkNotNullArgument(parentId, "Parent group Id");
      checkNotNullArgument(memberId, "Member user Id");

      getRepository().createRelationship(getInvocationContext(), createIdentityObjectForGroupId(parentId), createIdentityObjectForUserName(memberId), MEMBER, null, true);

   }

   public void disassociateGroups(Collection<Group> parents, Collection<Group> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<Group> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            Group member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null);
         }
      }
   }

   public void disassociateGroups(Group parent, Collection<Group> members) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent");

      disassociateGroups(Arrays.asList(parent), members);
   }

   public void disassociateGroupsByIds(Collection<String> parents, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<String> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         String parent = parentsIterator.next();

         for (Iterator<String> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            String member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObjectForGroupId(parent), createIdentityObjectForGroupId(member), MEMBER, null);
         }
      }
   }

   public void disassociateGroupsByIds(String parent, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent Id");

      disassociateGroupsByIds(Arrays.asList(parent), members);
   }

   public void disassociateUsers(Collection<Group> parents, Collection<User> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");

      for (Iterator<Group> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         Group parent = parentsIterator.next();

         for (Iterator<User> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            User member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER, null);
         }
      }

   }

   public void disassociateUsers(Group parent, Collection<User> members) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent group");

      disassociateUsers(Arrays.asList(parent), members);
   }

   public void disassociateUsersByIds(Collection<String> parents, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");


      for (Iterator<String> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         String parent = parentsIterator.next();

         for (Iterator<String> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            String member = membersIterator.next();

            getRepository().removeRelationship(getInvocationContext(), createIdentityObjectForGroupId(parent), createIdentityObjectForUserName(member), MEMBER, null);
         }
      }
   }

   public void disassociateUsersByIds(String parent, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent Id");

      disassociateUsersByIds(Arrays.asList(parent), members);

   }

   public <G extends IdentityType, I extends IdentityType> boolean isAssociated(Collection<G> parents, Collection<I> members) throws IdentityException
   {
      //TODO: maybe IdentityStore should have isRelationshipPresent method to improve this?

      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");


      for (Iterator<G> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         IdentityType parent = parentsIterator.next();

         for (Iterator<I> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            IdentityType member = membersIterator.next();

            Collection<IdentityObjectRelationship> relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER);

            if (relationships.size() == 0)
            {
               return false;
            }
         }
      }

      return true;
   }

   public boolean isAssociatedByIds(Collection<String> parents, Collection<String> members) throws IdentityException
   {
      checkNotNullArgument(parents, "parents");
      checkNotNullArgument(members, "members");


      for (Iterator<String> parentsIterator = parents.iterator(); parentsIterator.hasNext();)
      {
         String parent = parentsIterator.next();

         for (Iterator<String> membersIterator = members.iterator(); membersIterator.hasNext();)
         {
            String member = membersIterator.next();

            Collection<IdentityObjectRelationship> relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER);

            if (relationships.size() == 0)
            {
               return false;
            }
         }
      }
      return true;
   }

   public <G extends IdentityType, I extends IdentityType> boolean isAssociated(G parent, I member) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent IdentityType");
      checkNotNullArgument(member, "Member IdentityType");


      Set<G> parents = new HashSet<G>();
      parents.add(parent);
      Set<I> members = new HashSet<I>();
      members.add(member);

      return isAssociated(parents, members);
   }

   public boolean isAssociatedByIds(String parent, String member) throws IdentityException
   {
      checkNotNullArgument(parent, "Parent Id");
      checkNotNullArgument(member, "Member Id");


      Collection<IdentityObjectRelationship> relationships = getRepository().resolveRelationships(getInvocationContext(), createIdentityObject(parent), createIdentityObject(member), MEMBER);

      if (relationships.size() == 0)
      {
         return false;
      }

      return true;
   }

   public Collection<Group> findAssociatedGroups(Group group, String groupType, boolean parent, boolean inherited, IdentitySearchCriteria controls) throws IdentityException
   {

      checkNotNullArgument(group, "Group");
//      checkNotNullArgument(groupType, "Group type");

      List<Group> identities = new LinkedList<Group>();

      IdentityObjectType iot = groupType != null ? getIdentityObjectType(groupType) : null;

      //TODO Handle inherited
      if (inherited)
      {
         throw new NotYetImplementedException("Support for 'inherited' argument is not yet implemented. Please use 'false' value for now");
      }

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), MEMBER, parent, convertSearchControls(controls));

      for (IdentityObject io : ios)
      {
         if (iot == null || io.getIdentityType().getName().equals(iot.getName()))
         {
            identities.add(createGroup(io));
         }
      }

      return identities;

   }

   public Collection<Group> findAssociatedGroups(String groupId, String groupType, boolean parent, boolean inherited, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group Id");
//      checkNotNullArgument(groupType, "Group type");

      Group group = createGroupFromId(groupId);

      return findAssociatedGroups(group, groupType, parent, inherited, controls);
   }

   public Collection<Group> findAssociatedGroups(Group group, String groupType, boolean parent, boolean inherited) throws IdentityException
   {
      checkNotNullArgument(group, "Group");
//      checkNotNullArgument(groupType, "Group type");

      return findAssociatedGroups(group, groupType, parent, inherited, null);
   }

   public Collection<Group> findAssociatedGroups(User identity, String groupType, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(identity, "User");
      //checkNotNullArgument(groupType, "Group type");

      List<Group> identities = new LinkedList<Group>();

      IdentityObjectType iot = groupType != null ? getIdentityObjectType(groupType) : null;

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(identity), MEMBER, false, convertSearchControls(controls));

      for (IdentityObject io : ios)
      {
         if (iot == null || io.getIdentityType().getName().equals(iot.getName()))
         {
            identities.add(createGroup(io));
         }
      }

      return identities;
   }

   public Collection<Group> findAssociatedGroups(String userName, String groupType, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");
      //checkNotNullArgument(groupType, "Group type");

      User user = createUserFromId(userName);

      return findAssociatedGroups(user, groupType, controls);
   }

   public Collection<Group> findAssociatedGroups(User identity, String groupType) throws IdentityException
   {
      checkNotNullArgument(identity, "User");
      //checkNotNullArgument(groupType, "Group type");

      return findAssociatedGroups(identity, groupType, null);
   }

   public Collection<Group> findAssociatedGroups(User identity, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(identity, "User");

      List<Group> identities = new LinkedList<Group>();

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(identity), MEMBER, false, convertSearchControls(controls));


      String userTypeName = getUserObjectType().getName();

      for (IdentityObject io : ios)
      {

         // Filter out users
         if (!io.getIdentityType().getName().equals(userTypeName))
            identities.add(createGroup(io));
      }

      return identities;
   }

   public Collection<Group> findAssociatedGroups(String userName, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");

      User user = createUserFromId(userName);

      return findAssociatedGroups(user, controls);
   }

   public Collection<Group> findAssociatedGroups(User identity) throws IdentityException
   {
      checkNotNullArgument(identity, "User");

      return findAssociatedGroups(identity, (IdentitySearchCriteria)null);
   }

   public Collection<User> findAssociatedUsers(Group group, boolean inherited, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      List<User> identities = new LinkedList<User>();

      //TODO Handle inherited
      if (inherited)
      {
         throw new NotYetImplementedException("Support for 'inherited' argument is not yet implemented. Please use 'false' value for now");
      }

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), MEMBER, true, convertSearchControls(controls));

      String userTypeName = getUserObjectType().getName();

      for (IdentityObject io : ios)
      {
         //Filter out groups
         if (io.getIdentityType().getName().equals(userTypeName))
         {
            identities.add(createUser(io));
         }
      }

      return identities;
   }

   public Collection<User> findAssociatedUsers(String groupId, boolean inherited, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group Id");

      Group group = createGroupFromId(groupId);

      return findAssociatedUsers(group, inherited, controls);
   }

   public Collection<User> findAssociatedUsers(Group group, boolean inherited) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      return findAssociatedUsers(group, inherited, null);
   }

   public Collection<Group> findRelatedGroups(User user, String groupType, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(user, "User");

      List<Group> identities = new LinkedList<Group>();

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(user), null, false, convertSearchControls(controls));


      String userTypeName = getUserObjectType().getName();

      for (IdentityObject io : ios)
      {
         // Filter out users
         if (!io.getIdentityType().getName().equals(userTypeName))
            identities.add(createGroup(io));
      }

      return identities;
   }

   public Collection<Group> findRelatedGroups(String userName, String groupType, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(userName, "User name");

      User user = createUserFromId(userName);

      return findRelatedGroups(user, groupType, controls);
   }

   public Collection<User> findRelatedUsers(Group group, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(group, "Group");

      List<User> identities = new LinkedList<User>();

      Collection<IdentityObject> ios = getRepository().findIdentityObject(getInvocationContext(), createIdentityObject(group), null, true, convertSearchControls(controls));

      String userTypeName = getUserObjectType().getName();

      for (IdentityObject io : ios)
      {
         if (io.getIdentityType().getName().equals(userTypeName))
         {
            User user = createUser(io);

            if (!identities.contains(user))
            {
               identities.add(createUser(io));
            }
         }
      }

      return identities;
   }

   public Collection<User> findRelatedUsers(String groupId, IdentitySearchCriteria controls) throws IdentityException
   {
      checkNotNullArgument(groupId, "Group Id");

      Group group = createGroupFromId(groupId);

      return findRelatedUsers(group, controls);
   }
}
