/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.tools.arquillian.extension;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.arquillian.config.descriptor.api.ArquillianDescriptor;
import org.jboss.arquillian.container.spi.event.container.AfterStart;
import org.jboss.arquillian.container.spi.event.container.AfterStop;
import org.jboss.arquillian.container.spi.event.container.BeforeStart;
import org.jboss.arquillian.container.spi.event.container.BeforeStop;
import org.jboss.arquillian.core.api.Instance;
import org.jboss.arquillian.core.api.annotation.Inject;
import org.jboss.arquillian.core.api.annotation.Observes;
import org.jboss.as.arquillian.container.ManagementClient;

/**
 * ServerConfiguration class that manages the functionality of extension of the same name.
 * 
 * @author ochaloup
 */
public class ServerConfiguration {
    private static final Logger log = Logger.getLogger(ServerConfiguration.class);   
    private List<ServerConfigurationDefinition> definitions;

    @Inject
    private Instance<ManagementClient> managementClient;
    
    /**
     * The constructor servers for defining the "plugins" that will be run on arquillian hooks.
     */
    public ServerConfiguration() {
        definitions = new ArrayList<ServerConfigurationDefinition>();
        
        // For EAP5
        if(ArquillianUtils.getEAPVersion() == 5) {
            // currently nothing
        }
        // For EAP6
        if(ArquillianUtils.getEAPVersion() == 6) {
            definitions.add(new EAP6TestProperties());
            definitions.add(new EAP6BackupAndRestore());
            definitions.add(new EAP6UserAuthConfiguration());
            definitions.add(new EAP6LoggingLevelSetting(EAP6LoggingLevelSetting.LoggerLevel.DEBUG));
        }
    }
    
    /**
     * On container start
     */
    public void beforeStart(@Observes BeforeStart event, ArquillianDescriptor descriptor) throws IOException {
        if(isNotConfig())
            return;
        
        for(ServerConfigurationDefinition def: definitions) {
            log.debug("Running configuration " + def + " on beforeStart");
            def.beforeStart(managementClient.get(), descriptor);
        }
    }
    
    /**
     * After container started
     */
    public void afterStart(@Observes AfterStart event, ArquillianDescriptor descriptor) throws IOException {
        if(isNotConfig())
            return;
        
        for(ServerConfigurationDefinition def: definitions) {
            log.debug("Running configuration " + def + " on afterStart");
            def.afterStart(managementClient.get(), descriptor);
        }
    }
    
    /**
     * Before container shutdown
     */
    public void beforeShutdown(@Observes BeforeStop event, ArquillianDescriptor descriptor) throws IOException {
        if(isNotConfig())
            return;
        
        for(ServerConfigurationDefinition def: definitions) {
            log.debug("Running configuration " + def + " on beforeShutdown");
            def.beforeShutdown(managementClient.get(), descriptor);
        }
    }

    /**
     * On container stop
     */
    public void afterShutdown(@Observes AfterStop event, ArquillianDescriptor descriptor) throws IOException {
        if(isNotConfig())
            return;
        
        for(ServerConfigurationDefinition def: definitions) {
            log.debug("Running configuration " + def + " on afterShutdown");
            def.afterShutdown(managementClient.get(), descriptor);
        }
    }
    
    private boolean isNotConfig() {
        return Boolean.valueOf(System.getProperty("dont.change.config", "false"));
    }
}
