/*
* JBoss, Home of Professional Open Source.
* Copyright 2011, Red Hat, Inc., and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.qa.ejb.backwardscompatibility.manualmode.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import java.util.concurrent.Future;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.jboss.arquillian.container.test.api.ContainerController;
import org.jboss.arquillian.container.test.api.Deployer;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.container.test.api.TargetsContainer;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.ejb.client.ContextSelector;
import org.jboss.ejb.client.EJBClientConfiguration;
import org.jboss.ejb.client.EJBClientContext;
import org.jboss.ejb.client.PropertiesBasedEJBClientConfiguration;
import org.jboss.ejb.client.remoting.ConfigBasedEJBClientContextSelector;
import org.jboss.logging.Logger;
import org.jboss.qa.ejb.backwardscompatibility.shared.SendRequest;
import org.jboss.qa.ejb.backwardscompatibility.shared.SendRequestRemote;
import org.jboss.qa.ejb.backwardscompatibility.shared.StaticInfoProvider;
import org.jboss.qa.ejb.backwardscompatibility.shared.TestHelper;
import org.jboss.qa.ejb.backwardscompatibility.shared.ejb3.StatefulBean;
import org.jboss.qa.ejb.backwardscompatibility.shared.ejb3.StatelessBean;
import org.jboss.qa.ejb.backwardscompatibility.shared.ejb31.AsyncBeanRemote;
import org.jboss.qa.ejb.backwardscompatibility.shared.ejb31.AsyncStatefulBean;
import org.jboss.qa.ejb.backwardscompatibility.shared.ejb31.AsyncStatelessBean;
import org.jboss.remoting3.Version;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Testing ssl connection of remote ejb client. EAP6 only.
 *
 * @author Jan Martiska
 */
@RunWith(Arquillian.class)
@RunAsClient
public class SSLEJBRemoteClientTestCase {
    public static final String MODULE_NAME_STATELESS = "ssl-remote-ejb-client-test";
    public static final String MODULE_NAME_STATEFUL = "ssl-remote-ejb-client-test-stateful";
    public static final String MODULE_NAME_ASYNC = "ssl-remote-ejb-client-test-async";

    private static boolean serverConfigDone = false;

    private static ContextSelector<EJBClientContext> previousClientContextSelector;

    private static final String address = "localhost";
    private static final int portManagement = 9999;
    private static final String portRemotingSSL = "4448";

    private static final Logger log = Logger.getLogger(SSLEJBRemoteClientTestCase.class);

    private InitialContext ctx;

    @ArquillianResource
    private static ContainerController container;

    public static final String CLASSLOADER_SCAN_PROPERTY_NAME
            = "jboss.ejb.client.properties.skip.classloader.scan";

    @ArquillianResource
    private Deployer deployer;

    public static final String DEFAULT_JBOSSAS = "eap-manualmode";

    @Deployment(name = MODULE_NAME_STATELESS, managed = false, testable = false)
    @TargetsContainer(DEFAULT_JBOSSAS)
    public static JavaArchive deployStateless() {
        final JavaArchive archive = ShrinkWrap.create(JavaArchive.class, MODULE_NAME_STATELESS + ".jar");
        archive.addClasses(StatelessBean.class, SendRequest.class, SendRequestRemote.class,
                StaticInfoProvider.class);
        return archive;
    }

    @Deployment(name = MODULE_NAME_STATEFUL, managed = false, testable = false)
    @TargetsContainer(DEFAULT_JBOSSAS)
    public static JavaArchive deployStateful() {
        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, MODULE_NAME_STATEFUL + ".jar");
        jar.addClasses(StatefulBean.class, SendRequest.class, SendRequestRemote.class,
                StaticInfoProvider.class);
        return jar;
    }

    @Deployment(name = MODULE_NAME_ASYNC, managed = false, testable = false)
    @TargetsContainer(DEFAULT_JBOSSAS)
    public static JavaArchive deployAsync() {
        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, MODULE_NAME_ASYNC + ".jar");
        jar.addClasses(AsyncBeanRemote.class, AsyncStatefulBean.class, AsyncStatelessBean.class);
        return jar;
    }

    @BeforeClass
    public static void prepare() throws Exception {
        log.info("*** BEFORE CLASS ***");
        System.setProperty("dont.change.config", "true");
        log.info("*** javax.net.ssl.trustStore=" + System.getProperty("javax.net.ssl.trustStore"));
        log.info("*** javax.net.ssl.trustStorePassword=" + System
                .getProperty("javax.net.ssl.trustStorePassword"));
        log.info("*** javax.net.ssl.keyStore=" + System.getProperty("javax.net.ssl.keyStore"));
        log.info(
                "*** javax.net.ssl.keyStorePassword=" + System.getProperty("javax.net.ssl.keyStorePassword"));
        if (!TestHelper.getJndiLookupType().equalsIgnoreCase("ejb")) {
            System.setProperty(CLASSLOADER_SCAN_PROPERTY_NAME, "true");
        }
    }

    private static ContextSelector<EJBClientContext> setupEJBClientContextSelector() throws IOException {
        log.info("*** reading EJBClientContextSelector properties");
        // setup the selector
        final String clientPropertiesFile = "jboss-ejb-client.properties";

        final InputStream inputStream = SSLEJBRemoteClientTestCase.class.getClassLoader()
                .getResourceAsStream(clientPropertiesFile);
        if (inputStream == null) {
            throw new IllegalStateException("Could not find " + clientPropertiesFile + " in classpath");
        }
        final Properties properties = new Properties();
        properties.load(inputStream);
        final EJBClientConfiguration ejbClientConfiguration = new PropertiesBasedEJBClientConfiguration(
                properties);
        log.info("*** creating EJBClientContextSelector");
        final ConfigBasedEJBClientContextSelector selector = new ConfigBasedEJBClientContextSelector(
                ejbClientConfiguration);
        log.info("*** applying EJBClientContextSelector");
        return EJBClientContext.setSelector(selector);
    }


    @Before
    public void prepareServerOnce() throws Exception {
        try {
            if (!serverConfigDone) {
                // prepare server config and then restart
                container.start(DEFAULT_JBOSSAS);
                ManagementClient managementClient;
                final ModelControllerClient client = ModelControllerClient.Factory.create(address,
                        portManagement);
                managementClient = new ManagementClient(client, address, portManagement);
                new EAP6LoggingLevelSettingTool(EAP6LoggingLevelSettingTool.LoggerLevel.DEBUG).afterStart(managementClient, null);
                if (!SSLRealmSetupTool.isAlreadyConfigured(managementClient)) {
                    SSLRealmSetupTool.setup(managementClient, container, DEFAULT_JBOSSAS);
                    managementClient = new ManagementClient(client, address, portManagement);
                    // write SSL realm config to output - debugging purposes
                    SSLRealmSetupTool.readSSLRealmConfig(managementClient);
                    serverConfigDone = true;
                } else {
                    log.info("*** server seems already configured from a previous testsuite run");
                }
            } else {
                log.info("*** skipping configuration routine, already done that");
            }
        } catch(Exception e) {
            e.printStackTrace();
            Assert.fail("Unable to setup SSL connector realm");
        }
    }

    @AfterClass
    public static void tearDown() throws Exception {
        log.info("CLEANUP starting");
        if (previousClientContextSelector != null) {
            EJBClientContext.setSelector(previousClientContextSelector);
        }
        final ModelControllerClient client = ModelControllerClient.Factory.create(address, portManagement);
        final ManagementClient mClient = new ManagementClient(client, address, portManagement);
        SSLRealmSetupTool.tearDown(mClient, container);
        mClient.close();
        System.setProperty("dont.change.config", "false");
    }

    private Properties getEjbClientContextProperties() throws IOException {
        log.info("*********************** JBOSS REMOTING VERSION: " + Version.VERSION);
        String jndiLookupType = TestHelper.getJndiLookupType();
        Properties env = new Properties();
        if (jndiLookupType.equalsIgnoreCase("remote")) {
            env.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.naming.remote.client.InitialContextFactory");
//            env.put(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
            env.put(Context.PROVIDER_URL, "remote://127.0.0.1:4448");
            env.put("jboss.naming.client.ejb.context", true);
            env.put("jboss.naming.client.connect.options.org.xnio.Options.SASL_POLICY_NOPLAINTEXT", "false");
            env.put("jboss.naming.client.connect.options.org.xnio.Options.SSL_STARTTLS", "true");
            env.put("jboss.naming.client.connect.options.org.xnio.Options.SSL_ENABLED", "true");
            env.put("jboss.naming.client.connect.options.org.xnio.Options.SASL_POLICY_NOANONYMOUS", "false");
            env.put("jboss.naming.client.connect.options.org.xnio.Options.SASL_DISALLOWED_MECHANISMS",
                    "false");
            env.put("jboss.naming.client.remote.connectionprovider.create.options.org.xnio.Options.SSL_ENABLED",
                    "true");
        } else if (jndiLookupType.equalsIgnoreCase("ejb")) {
            env.setProperty(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
            env.put("jboss.naming.client.ejb.context", true);
        } else if (jndiLookupType.equalsIgnoreCase("ejb_mapbased")) {
            env.setProperty(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
            env.put("org.jboss.ejb.client.scoped.context", true);
            env.put("endpoint.name", "client-endpoint");
            env.put("remote.connections", "main");
            env.put("remote.connection.main.host", address);
            env.put("remote.connection.main.port", portRemotingSSL);
            env.put("remote.connection.main.connect.options.org.xnio.Options.SASL_POLICY_NOANONYMOUS",
                    "false");
            env.put("remote.connection.main.connect.options.org.xnio.Options.SSL_ENABLED", "true");
            env.put("remote.connection.main.connect.options.org.xnio.Options.SSL_STARTTLS", "true");
            env.put("remote.connection.main.connect.options.org.xnio.Options.SASL_POLICY_NOPLAINTEXT",
                    "true");
        } else {
            throw new RuntimeException("Unsupported JNDI lookup type: " + jndiLookupType);
        }
        return env;
    }


    @Test
    public void testStatelessBeanByte() throws Exception {
        log.info("################ testStatelessBeanByte ##################");
        log.info("**** deploying application");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        byte response = bean.sendByte(Byte.parseByte("8"));
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", Byte.parseByte("8"), response);
        deployer.undeploy(MODULE_NAME_STATELESS);
    }


    @Test
    public void testStatelessBeanInt() throws Exception {
        log.info("################ testStatelessBeanInt ##################");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        try {
            bean.sendInt(50);
        } catch (Exception e) {
        }
        log.info("**** About to perform synchronous call on stateless bean");
        Integer response = bean.sendInt(50);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", 50L, (long)response);
        deployer.undeploy(MODULE_NAME_STATELESS);
    }

    @Test
    public void testStatefulBeanByte() throws Exception {
        log.info("################ testStatefulBeanByte ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("0");
        log.info("**** About to perform synchronous call on stateful bean");
        byte response = bean.sendByte(Byte.parseByte("8"));
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", Byte.parseByte("8"), response);
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatelessBeanStringMap() throws Exception {
        log.info("################ testStatelessBeanStringMap ##################");
        Map<Integer, String> map = new TreeMap<Integer, String>();
        map.put(1, "lala");
        map.put(2, "trala");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        Map<Integer, String> response = bean.sendStringMap(map);
        log.info("**** The answer is: " + response);
        Assert.assertTrue(response.get(1).startsWith("lala"));
        Assert.assertTrue(response.get(2).startsWith("trala"));
        deployer.undeploy(MODULE_NAME_STATELESS);
    }

    @Test
    public void testStatefulBeanStringMap() throws Exception {
        log.info("################ testStatefulBeanStringMap ##################");
        Map<Integer, String> map = new TreeMap<Integer, String>();
        map.put(1, "lala");
        map.put(2, "trala");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("0");

        log.info("**** About to perform synchronous call on stateful bean");
        Map<Integer, String> response = bean.sendStringMap(map);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", map.get(1), "lala");
        Assert.assertEquals("Remote invocation of EJB was not successful", map.get(2), "trala");
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatefulBeanInt() throws Exception {
        log.info("################ testStatefulBeanInt ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("3");
        log.info("**** About to perform synchronous call on stateful bean");
        Integer response = bean.sendInt(50);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", 53L, (long)response);
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatelessBeanString() throws Exception {
        log.info("################ testStatelessBeanString ##################");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        String response = bean.sendString("hi");
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", "hinull", response);
        deployer.undeploy(MODULE_NAME_STATELESS);
    }

    @Test
    public void testStatelessBeanBoolean() throws Exception {
        log.info("################ testStatelessBeanBoolean ##################");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        Boolean response = bean.sendBoolean(true);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", true, response);
        deployer.undeploy(MODULE_NAME_STATELESS);
    }

    @Test
    public void testStatefulBeanString() throws Exception {
        log.info("################ testStatefulBeanString ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("3");
        log.info("**** About to perform synchronous call on stateful bean");
        String response = bean.sendString("hi");
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", "hi3", response);
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatefulBeanBoolean() throws Exception {
        log.info("################ testStatefulBeanBoolean ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("3");
        log.info("**** About to perform synchronous call on stateful bean");
        Boolean response = bean.sendBoolean(true);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", true, response);
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatelessBeanDouble() throws Exception {
        log.info("################ testStatelessBeanDouble ##################");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        Double response = bean.sendDouble(15.0);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", (Double)15.0, response);
        deployer.undeploy(MODULE_NAME_STATELESS);
    }

    @Test
    public void testStatefulBeanDouble() throws Exception {
        log.info("################ testStatefulBeanDouble ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("3");
        log.info("**** About to perform synchronous call on stateful bean");
        Double response = bean.sendDouble(15.0);
        log.info("**** The answer is: " + response);
        Assert.assertEquals("Remote invocation of EJB was not successful", (Double)18.0, response);
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testStatelessBeanIntArray() throws Exception {
        log.info("################ testStatelessBeanIntArray ##################");
        log.info("**** deploying deployment with stateless beans");
        deployer.deploy(MODULE_NAME_STATELESS);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatelessBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATELESS, StatelessBean.class, SendRequestRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        int[] response = bean.sendIntArray(new int[] {3, 5, 8});
        log.info("**** The answer is: " + Arrays.toString(response));
        Assert.assertTrue("Remote invocation of EJB was not successful",
                Arrays.equals(new int[] {3, 5, 8}, response));
        deployer.undeploy(MODULE_NAME_STATELESS);

    }

    @Test
    public void testStatefulBeanIntArray() throws Exception {
        log.info("################ testStatefulBeanIntArray ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_STATEFUL);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        SendRequestRemote bean = (SendRequestRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_STATEFUL, StatefulBean.class, SendRequestRemote.class,
                        true));
        bean.setVariableParam("3");
        log.info("**** About to perform synchronous call on stateful bean");
        int[] response = bean.sendIntArray(new int[] {3, 5, 8});
        log.info("**** The answer is: " + Arrays.toString(response));
        Assert.assertTrue("Remote invocation of EJB was not successful",
                Arrays.equals(new int[] {6, 8, 11}, response));
        deployer.undeploy(MODULE_NAME_STATEFUL);
    }

    @Test
    public void testAsyncStatelessBeanInt() throws Exception {
        log.info("################ testAsyncStatelessBeanInt ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_ASYNC);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        AsyncBeanRemote bean = (AsyncBeanRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_ASYNC, AsyncStatelessBean.class, AsyncBeanRemote.class,
                        false));
        log.info("**** About to perform synchronous call on stateless bean");
        Future<Boolean> futureResponse = bean.futureMethod();
        boolean response = futureResponse.get();
        log.info("**** The answer is: " + response);
        Assert.assertTrue("Remote invocation of EJB was not successful", response);
        deployer.undeploy(MODULE_NAME_ASYNC);
    }

    @Test
    public void testAsyncStatefulBeanInt() throws Exception {
        log.info("################ testAsyncStatefulBeanInt ##################");
        log.info("**** deploying deployment with stateful beans");
        deployer.deploy(MODULE_NAME_ASYNC);
        log.info("**** creating InitialContext");
        ctx = new InitialContext(getEjbClientContextProperties());
        log.info("**** looking up StatefulBean through JNDI");
        AsyncBeanRemote bean = (AsyncBeanRemote)
                ctx.lookup(createJNDIName(MODULE_NAME_ASYNC, AsyncStatefulBean.class, AsyncBeanRemote.class,
                        true));

        log.info("**** About to perform synchronous call on stateful bean");
        Future<Boolean> futureResponse = bean.futureMethod();
        boolean response = futureResponse.get();
        log.info("**** The answer is: " + response);
        Assert.assertTrue("Remote invocation of EJB was not successful", response);
        deployer.undeploy(MODULE_NAME_ASYNC);
    }

    public String createJNDIName(String moduleName, Class bean, Class beanRemoteInterface, boolean stateful) {
        if (TestHelper.getJndiLookupType().equalsIgnoreCase("remote")) {
            return moduleName + "/" + bean.getSimpleName() + "!" + beanRemoteInterface.getCanonicalName() + (
                    stateful ? "" : "");
        } else {
            return "ejb:/" + moduleName + "/" + bean.getSimpleName() + "!" + beanRemoteInterface
                    .getCanonicalName() + (stateful ? "?stateful" : "");
        }
    }

}