/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.qa.tools.arquillian.extension;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.arquillian.config.descriptor.api.ArquillianDescriptor;
import org.jboss.as.arquillian.container.ManagementClient;

/**
 * Backuping on start and restoring on stop.
 * 
 * @author ochaloup
 */
public class EAP6BackupAndRestore implements ServerConfigurationDefinition {
    private static final Logger log = Logger.getLogger(EAP6BackupAndRestore.class);
        
    // Files that were backuped for being possible to restore them
    private List<File> backupedFiles;
    
    
    @Override
    public void beforeStart(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        doBackup(descriptor);
    }

    @Override
    public void afterShutdown(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        doRestore();
    }
    
    @Override
    public void afterStart(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        // nothing to do
    }

    @Override
    public void beforeShutdown(ManagementClient managementClient, ArquillianDescriptor descriptor) {
        // nothing to do
    }
    

    private void doBackup(ArquillianDescriptor descriptor) {
        // We will BACKUP: 
        // 1. eap xml config files
        backupedFiles = ArquillianUtilsEAP6.getConfigFiles(descriptor);
        //  2. some other files (e.g. properties)
        List<File> otherFiles = ArquillianUtilsEAP6.getFilesInConfigDir(descriptor, 
                EAP6UserAuthConfiguration.ROLES_PROPERTIES_FILENAME, EAP6UserAuthConfiguration.USER_PROPERTIES_FILENAME);
        backupedFiles.addAll(otherFiles);
        
        try {
            for(File fileToBackup: backupedFiles) {
                File backupFile = constructBackupFileName(fileToBackup);
                ArquillianUtils.copyFile(fileToBackup, backupFile);
                log.debug("Backup file " + backupFile + " was created by copying from " + fileToBackup);
            }
        } catch (IOException ioe) {
            throw new RuntimeException("Can't copy file for backup: " + backupedFiles, ioe);
        }
    }
    
    private void doRestore() {
        for(File fileToRestore: backupedFiles) {
            File backupedFile = constructBackupFileName(fileToRestore);
            // trying to restore the backup
            try {
                ArquillianUtils.copyFile(backupedFile, fileToRestore);
                log.debug("File " + fileToRestore + " was restored from " + backupedFile);
            } catch (IOException ioe) {
                throw new RuntimeException("Can't copy " + backupedFiles, ioe);
            }
            // trying to delete the .bkp file - not needed anymore
            backupedFile.delete();
            log.debug("Backup file " + backupedFile + " was deleted after restoring the original content in " + fileToRestore);
        } 
    }
   
    /**
     * Returns file named with .bkp suffix.
     */
    private File constructBackupFileName(File file) {
    	String backupSuffix = ".test.bkp";
    	return new File(file.getAbsolutePath() + backupSuffix);
    }
}
