/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.internal;

// $Id: BasicBridge.java 89047 2009-05-19 14:05:01Z thomas.diesler@jboss.com $

import java.util.Properties;

import org.jboss.osgi.husky.Bridge;
import org.jboss.osgi.husky.Invoker;

/**
 * A basic {@link Bridge} implementation that can be associated
 * with an arbitrary {@link Invoker}.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public class BasicBridge extends AbstractBridge
{
   private Properties props;
   private Invoker invoker;

   /**
    * Construct a basic {@link Bridge} with the given properties.
    * 
    *  The properties would typically contain the name of the 
    *  associated invoker under the key of {@link Invoker}. 
    */
   public BasicBridge(Properties props)
   {
      this.props = props;
   }

   /**
    * Get the invoker associated with this {@link Bridge}.
    * 
    * If the invoker was not explicitly set, this method will load the 
    * invoker from the class name given by the {@link Invoker} property.
    * 
    * If still not found it will load the invoker from the class name 
    * given by the {@link Invoker} system property.
    * 
    * @throws RuntimeException if the {@link Invoker} cannot be loaded. 
    */
   public Invoker getInvoker()
   {
      if (invoker == null)
      {
         String className = props.getProperty(Invoker.class.getName());
         if (className == null)
            className = System.getProperty(Invoker.class.getName());

         if (className == null)
            throw new IllegalStateException("Cannot obtain value for invoker property: " + Invoker.class.getName());

         invoker = (Invoker)Util.loadInstance(className, props);
      }
      return invoker;
   }

   /**
    * Set the associated invoker
    */
   public void setInvoker(Invoker invoker)
   {
      this.invoker = invoker;
   }
}
