/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky;


// $Id$

/**
 * A Bridge intercepts a test and delegates the execution 
 * to the same (or another) test in and isolated test environment.
 * 
 * An isolated test environment is one that does not have the same 
 * class loading space as the test itself. Isolated test environments 
 * are for example 
 * 
 * <ul>
 * <li>Remote container environment</li>
 * <li>Embedded OSGi Framework</li> 
 * </ul>
 * 
 * Details on test method mapping is given with the individual 
 * run methods.
 * 
 * The test {@link Request} is fieled through an {@link Invoker}
 * that that returns a {@link Response} for the given test run. 
 * 
 * Invokers may be arbitarily complex. Local 'in proccess' invokers are
 * possible just as well as remote invokers. 
 * 
 * <pre>
 * void testMethod 
 * {
 *    if(isNotInContainer())
 *    {
 *       BridgeFactory.getBridge().run();
 *       return;
 *    }
 *    else
 *    {
 *       // in container test logic 
 *       ...
 *    }
 * }
 * </pre>
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public interface Bridge
{
   /**
    * Get the caller's class
    */
   Class<?> getCallerClass();
   
   /**
    * Get the caller's method
    */
   String getCallerMethod();
   
   /**
    * Get the caller's concatination of class name and method. 
    */
   String getCallerFrame();
   
   /**
    * Intercepts a test and delegates the execution to the
    * same test in the same test class.
    * 
    * The main use case is that of the same test class beeing deployed 
    * in the isolated test environment as well as run from the client test runner. 
    */
   void run();

   /**
    * Intercepts a test and delegates the execution to the
    * same test in a different test class.
    * 
    * The main use case is that of one client side test class with the same 
    * set of test methods as another test class that runs in the isolated test environment.
    */
   void run(Class<?> testClass);

   /**
    * Intercepts a test and delegates the execution to another test 
    * in a different test class.
    * 
    * The main use case is that of free mapping between client side tests and tests
    * that run in the isolated test environment.
    */
   void run(Class<?> testClass, String testMethod);

   /**
    * Get the invoker that is associated with this bridge.
    */
   Invoker getInvoker();

   /**
    * Set the invoker that is associated with this bridge.
    */
   void setInvoker(Invoker invoker);
}
