/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.internal;

// $Id$

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import junit.framework.AssertionFailedError;

import org.jboss.osgi.husky.Bridge;
import org.jboss.osgi.husky.Failure;
import org.jboss.osgi.husky.Response;

/**
 * An abstract {@link Bridge} implementation. 
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public abstract class AbstractBridge implements Bridge
{
   private Response response;
   
   public void run()
   {
      String callerFrame = getCallerFrame();
      run(getCallerClass(callerFrame), getCallerMethod(callerFrame));
   }

   public void run(Class<?> testClass)
   {
      String callerFrame = getCallerFrame();
      run(testClass, getCallerMethod(callerFrame));
   }

   public void run(Class<?> testClass, String testMethod)
   {
      if (response == null)
      {
         BasicRequest request = new BasicRequest(testClass.getName(), testMethod);
         response = getInvoker().invoke(request);
      }

      List<Failure> failures = response.getFailures();
      for (Failure failure : failures)
      {
         String failedClass = failure.getClassName();
         if (failedClass.equals(testClass.getName()))
         {
            String failedMethod = failure.getMethodName();
            if (failedMethod == null || failedMethod.equals(testMethod))
            {
               Throwable failureCause = failure.getException();
               if (failureCause instanceof RuntimeException)
                  throw (RuntimeException)failureCause;
               if (failureCause instanceof Error)
                  throw (Error)failureCause;

               AssertionFailedError error = new AssertionFailedError(failure.getMessage());
               error.initCause(failureCause);
               throw error;
            }
         }
      }
   }

   public Class<?> getCallerClass()
   {
      String callerFrame = getCallerFrame();
      String className = callerFrame.substring(0, callerFrame.lastIndexOf('.'));
      return Util.loadClass(className);
   }
   
   public String getCallerMethod()
   {
      String callerFrame = getCallerFrame();
      return callerFrame.substring(callerFrame.lastIndexOf('.') + 1);
   }
   
   public String getCallerFrame()
   {
      String callerFrame = null;
      try
      {
         throw new Exception();
      }
      catch (Exception ex)
      {
         StringWriter sw = new StringWriter();
         ex.printStackTrace(new PrintWriter(sw));

         try
         {
            BufferedReader br = new BufferedReader(new StringReader(sw.toString()));

            // line 1 is java.lang.Exception
            // line 2 is this run method
            String line = br.readLine();
            line = br.readLine().trim();
            
            String bridgePackage = Bridge.class.getPackage().getName();
            while (line.startsWith("at " + bridgePackage))
            {
               line = br.readLine().trim();
            }
            callerFrame = line.substring(3, line.indexOf('('));
         }
         catch (IOException ioex)
         {
            // ignore
         }
      }
      return callerFrame;
   }

   private Class<?> getCallerClass(String callerFrame)
   {
      String className = callerFrame.substring(0, callerFrame.lastIndexOf('.'));
      return Util.loadClass(className);
   }
   
   private String getCallerMethod(String callerFrame)
   {
      return callerFrame.substring(callerFrame.lastIndexOf('.') + 1);
   }
}
