/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.internal;

// $Id$

import java.util.ArrayList;

import javax.management.MBeanServer;
import javax.management.MBeanServerConnection;
import javax.management.MBeanServerFactory;

import org.jboss.logging.Logger;
import org.jboss.osgi.husky.Invoker;
import org.jboss.osgi.husky.Request;
import org.jboss.osgi.husky.Response;

/**
 * An {@link Invoker} suitable for OSGi bundle testing.
 * 
 * It delegates the invocation to 
 * 
 * #1 {@link SocketInvocation}
 * #2 {@link JMXInvocation}
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public class AbstractInvoker implements Invoker
{
   // Provide logging
   private static final Logger log = Logger.getLogger(AbstractInvoker.class);
   
   public Response invoke(Request req)
   {
      log.debug("Start invoke: " + req);
      
      Response res = null;
      try
      {
         MBeanServerConnection server = getMBeanServer();
         if (SocketInvocation.isRemoteConnection())
         {
            res = new SocketInvocation().invoke(req);
         }
         else 
         {
            res = new JMXInvocation().invoke(server, req);
         }
         return res;
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception ex)
      {
         throw new IllegalStateException("Cannot invoke connector", ex);
      }
      finally
      {
         log.debug("End invoke: " + res);
      }
   }

   private MBeanServerConnection getMBeanServer()
   {
      ArrayList<MBeanServer> serverArr = MBeanServerFactory.findMBeanServer(null);
      if (serverArr.size() > 1)
         throw new IllegalStateException("Multiple MBeanServer instances not supported");

      MBeanServer server = null;
      if (serverArr.size() == 1)
         server = serverArr.get(0);

      if (server == null)
         server = MBeanServerFactory.createMBeanServer();

      return server;
   }

}
