/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.microcontainer.internal;

//$Id: MicrocontainerServiceImpl.java 89718 2009-06-03 09:37:53Z thomas.diesler@jboss.com $

import static org.osgi.framework.Constants.OBJECTCLASS;

import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import javax.management.MBeanServer;

import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.osgi.common.log.LogServiceTracker;
import org.jboss.osgi.integration.jbossas.AbstractMicrocontainerService;
import org.jboss.osgi.spi.management.MicrocontainerServiceMBean;
import org.jboss.osgi.spi.service.DeployerService;
import org.jboss.osgi.spi.service.MicrocontainerService;
import org.jboss.osgi.spi.service.XMLBindingService;
import org.jboss.virtual.VFS;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.log.LogService;
import org.osgi.util.tracker.ServiceTracker;

/**
 * An implementation of the {@link DeployerService} and {@link MicrocontainerService}
 * which is installed as a bundle
 * 
 * @author thomas.diesler@jboss.com
 * @since 23-Jan-2009
 */
public class MicrocontainerServiceImpl extends AbstractMicrocontainerService implements MicrocontainerServiceMBean
{
   private BundleContext context;
   private EmbeddedBeansDeployer deployer;
   private LogService log;

   public MicrocontainerServiceImpl(BundleContext context)
   {
      this.context = context;
      this.log = new LogServiceTracker(context);

      VFS.init();

      deployer = new EmbeddedBeansDeployer(context);
   }

   public Kernel getKernel()
   {
      return deployer.getKernel();
   }

   @Override
   public void logDebug(String message)
   {
      log.log(LogService.LOG_DEBUG, message);
   }

   @Override
   public void logWarning(String message, Exception ex)
   {
      log.log(LogService.LOG_WARNING, message, ex);
   }

   void start()
   {
      final MicrocontainerService mcServiceImpl = this;

      final Kernel kernel = getKernel();
      final KernelController controller = kernel.getController();

      // Preregister some beans
      installBean(controller, BEAN_SYSTEM_BUNDLE_CONTEXT, context);
      installBean(controller, BEAN_KERNEL, kernel);
      installBean(controller, BEAN_KERNEL_CONTROLLER, controller);

      // Setup a filter to track both services
      Filter filter;
      try
      {
         String jmxService = "(" + OBJECTCLASS + "=" + MBeanServer.class.getName() + ")";
         String xmlService = "(" + OBJECTCLASS + "=" + XMLBindingService.class.getName() + ")";
         filter = context.createFilter("(|" + jmxService + xmlService + ")");
      }
      catch (InvalidSyntaxException ex)
      {
         throw new IllegalArgumentException("Invalid filter syntax");
      }

      // Register the MicrocontainerServiceMBean
      ServiceTracker tracker = new ServiceTracker(context, filter, null)
      {
         XMLBindingService xmlService;
         MBeanServer mbeanServer;
         
         @Override
         public Object addingService(ServiceReference sref)
         {
            Object service = super.addingService(sref);
            
            List<String> classList = Arrays.asList((String[])sref.getProperty(OBJECTCLASS));
            if (classList.contains(MBeanServer.class.getName()))
               mbeanServer = (MBeanServer)service;

            if (classList.contains(XMLBindingService.class.getName()))
               xmlService = (XMLBindingService)service;

            if (mbeanServer != null && xmlService != null)
            {
               // Install the MBeanServer
               installBean(controller, BEAN_MBEAN_SERVER, mbeanServer);

               // Deploy the deployer beans
               URL deployersURL = context.getBundle().getResource("META-INF/base-deployers-beans.xml");
               deployer.deploy(deployersURL);

               // Register the MicrocontainerService
               log.log(LogService.LOG_DEBUG, "Register MicrocontainerService");
               context.registerService(MicrocontainerService.class.getName(), mcServiceImpl, null);

               // Register the DeployerService
               Properties props = new Properties();
               props.setProperty("provider", "microcontainer");
               log.log(LogService.LOG_DEBUG, "Register DeployerService");
               context.registerService(DeployerService.class.getName(), mcServiceImpl, props);

               // Register the MicrocontainerServiceMBean
               registerMicrocontainerServiceMBean(mbeanServer);
            }

            return service;
         }
      };
      tracker.open();
   }

   void stop()
   {
      ServiceReference sref = context.getServiceReference(MBeanServer.class.getName());
      if (sref != null)
      {
         MBeanServer mbeanServer = (MBeanServer)context.getService(sref);
         unregisterMicrocontainerServiceMBean(mbeanServer);
      }
   }
}