/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.container;

//$Id: BeanManager.java 90742 2009-07-01 10:51:48Z thomas.diesler@jboss.com $

import org.jboss.beans.metadata.plugins.AbstractValueMetaData;
import org.jboss.beans.metadata.plugins.builder.BeanMetaDataBuilderFactory;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.reflect.BeanMetadataImpl;
import org.jboss.osgi.blueprint.reflect.ReferenceMetadataImpl;
import org.jboss.reflect.spi.TypeInfo;
import org.osgi.service.blueprint.container.BlueprintContainer;
import org.osgi.service.blueprint.reflect.BeanMetadata;
import org.osgi.service.blueprint.reflect.BeanProperty;
import org.osgi.service.blueprint.reflect.Metadata;
import org.osgi.service.blueprint.reflect.ReferenceMetadata;
import org.osgi.util.tracker.ServiceTracker;

/**
 * A bean manager provides regular Java objects as component
 * instances. It has the following features:
 * 
 * • Construction via class name, static factory method, or a factory
 * method on a target. A target is a reference to a top level or inlined,
 * manager of type bean or service reference, or a referral to a top level
 * manager.
 * 
 * • Can have arguments for a constructor or factory method.
 * 
 * • Has properties that are configured.
 * 
 * • Manages a singleton or creates objects on demand depending on its scope.
 * 
 * • Life cycle callbacks for end of initialization and destruction.
 *  
 * @author thomas.diesler@jboss.com
 * @since 17-Jun-2009
 */
public class BeanManager extends AbstractManager
{
   private BeanMetadataImpl bpBeanMetadata;
   private BeanMetaData mcBeanMetadata;

   public BeanManager(BlueprintContext context, BlueprintContainer container, BeanMetadata compMetadata)
   {
      super(context, container, compMetadata);
      this.bpBeanMetadata = (BeanMetadataImpl)compMetadata;
   }

   @Override
   public void install()
   {
      super.install();
      
      // Install the kernel bean
      installKernelBean(getMicrocontainerBeanMetaData(bpBeanMetadata));
   }

   @Override
   public void shutdown()
   {
      if (mcBeanMetadata != null)
         uninstallKernelBean(mcBeanMetadata);

      super.shutdown();
   }

   public BeanMetaData getMicrocontainerBeanMetaData(BeanMetadataImpl compMetadata)
   {
      if (mcBeanMetadata == null)
      {
         String key = compMetadata.getKey();
         String clazz = compMetadata.getClassName();

         BeanMetaDataBuilder builder = BeanMetaDataBuilderFactory.createBuilder(key, clazz);
         builder.setClassLoader(((BlueprintContainerImpl)container).getClassLoader());
         mcBeanMetadata = builder.getBeanMetaData();
         
         for (BeanProperty prop : bpBeanMetadata.getProperties())
         {
            String name = prop.getName();
            Metadata value = prop.getValue();
            if (value instanceof BeanMetadata)
            {
               BeanMetadataImpl beanValue = (BeanMetadataImpl)value;
               BlueprintContainerImpl impl = (BlueprintContainerImpl)container;
               BeanManager beanManager = (BeanManager)impl.getComponentManager(beanValue.getKey());
               BeanMetaData valueMetaData = beanManager.getMicrocontainerBeanMetaData(beanValue);
               builder.addPropertyMetaData(name, valueMetaData);
            }
            else if (value instanceof ReferenceMetadata)
            {
               ReferenceMetadataImpl refValue = (ReferenceMetadataImpl)value;
               BlueprintContainerImpl impl = (BlueprintContainerImpl)container;
               ReferenceManager refManager = (ReferenceManager)impl.getComponentManager(refValue.getKey());
               ValueMetaData valueMetaData = new ServiceValueMetaData(refManager.getServiceTracker());
               builder.addPropertyMetaData(name, valueMetaData);
            }
            else
            {
               throw new IllegalStateException("Unsupported property value: " + value);
            }
         }
      }
      return mcBeanMetadata;
   }

   /**
    * A {@link ValueMetaData} implementation that gets its underlying value
    * from a {@link ServiceTracker}
    */
   private class ServiceValueMetaData extends AbstractValueMetaData
   {
      private static final long serialVersionUID = 1L;
      private ServiceTracker tracker;

      private ServiceValueMetaData(ServiceTracker tracker)
      {
         this.tracker = tracker;
         
         if (tracker == null)
            throw new IllegalArgumentException("Underlying ServiceTracker cannot be null");
      }

      @Override
      public Object getValue(TypeInfo info, ClassLoader cl)
      {
         return tracker.getService();
      }
   }
}