/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.parser.xb;

// $Id: TBlueprint.java 91345 2009-07-16 14:16:10Z thomas.diesler@jboss.com $

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlNsForm;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.namespace.QName;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.spi.NotImplementedException;
import org.jboss.xb.annotations.JBossXmlSchema;
import org.osgi.service.blueprint.container.NoSuchComponentException;

/**
 * The blueprint element is the top element.
 * 
 * <xsd:complexType name="Tblueprint">
 *  <xsd:sequence>
 *    <xsd:element name="description" type="Tdescription" minOccurs="0" />
 *    <xsd:element name="type-converters" type="Ttype-converters" minOccurs="0" maxOccurs="1" />
 *    <xsd:choice minOccurs="0" maxOccurs="unbounded">
 *      <xsd:element name="service" type="Tservice" />
 *      <xsd:element name="reference-list" type="Treference-list" />
 *      <xsd:element name="bean" type="Tbean" />
 *      <xsd:element name="reference" type="Treference" />
 *      <xsd:any namespace="##other" processContents="strict" />
 *    </xsd:choice>
 *  </xsd:sequence>
 *  <xsd:attribute name="default-activation" default="eager" type="Tactivation"/>
 *  <xsd:attribute name="default-timeout" type="Ttimeout" default="300000"/>
 *  <xsd:attribute name="default-availability" type="Tavailability" default="mandatory"/>
 *  <xsd:anyAttribute namespace="##other"  processContents="strict" />
 * </xsd:complexType>
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
@JBossXmlSchema(namespace = BlueprintContext.XMLNS_BLUEPRINT, elementFormDefault = XmlNsForm.QUALIFIED)
@XmlRootElement(name = "blueprint", namespace = BlueprintContext.XMLNS_BLUEPRINT)
@XmlType(name = "Tblueprint", namespace = BlueprintContext.XMLNS_BLUEPRINT, propOrder = { "description", "typeConverters", "components" })
public class TBlueprint
{
   protected String description;
   protected TTypeConverters typeConverters;
   protected List<TComponent> components;
   protected TActivation defaultActivation;
   protected BigInteger defaultTimeout;
   protected TAvailability defaultAvailability;
   private Map<QName, String> otherAttributes = new HashMap<QName, String>();

   public String getDescription()
   {
      return description;
   }

   public void setDescription(String value)
   {
      this.description = value;
   }

   public TTypeConverters getTypeConverters()
   {
      return typeConverters;
   }

   public void setTypeConverters(TTypeConverters value)
   {
      this.typeConverters = value;
   }

   @XmlElements( { 
      @XmlElement(name = "bean", namespace = BlueprintContext.XMLNS_BLUEPRINT, type = TBean.class),
      @XmlElement(name = "service", namespace = BlueprintContext.XMLNS_BLUEPRINT, type = TService.class),
      @XmlElement(name = "reference", namespace = BlueprintContext.XMLNS_BLUEPRINT, type = TReference.class),
      @XmlElement(name = "reference-list", namespace = BlueprintContext.XMLNS_BLUEPRINT, type = TReferenceList.class) })
   public List<TComponent> getComponents()
   {
      if (components == null)
         components = new ArrayList<TComponent>();

      return components;
   }

   public Set<String> getComponentIds()
   {
      Set<String> compIds = new LinkedHashSet<String>();
      for (TComponent comp : getComponents())
      {
         compIds.add(comp.getId());
      }
      return Collections.unmodifiableSet(compIds);
   }

   @SuppressWarnings("unchecked")
   public <T extends TComponent> List<T> getMetadata(Class<T> type)
   {
      List<T> compMetadata = new ArrayList<T>();
      for (TComponent comp : getComponents())
      {
         if (type.isAssignableFrom(comp.getClass()))
            compMetadata.add((T)comp);
      }
      return Collections.unmodifiableList(compMetadata);
   }

   public TComponent getComponentMetadata(String name)
   {
      if (name == null)
         throw new IllegalArgumentException("Cannot get metadata for id: null");

      TComponent compMetadata = null;
      for (TComponent comp : getComponents())
      {
         if (name.equals(comp.getId()))
         {
            compMetadata = comp;
            break;
         }
      }

      if (compMetadata == null)
         throw new NoSuchComponentException(name);

      return compMetadata;
   }

   @XmlAttribute
   public TActivation getDefaultActivation()
   {
      return defaultActivation;
   }

   public void setDefaultActivation(TActivation value)
   {
      this.defaultActivation = value;
   }

   @XmlAttribute
   public BigInteger getDefaultTimeout()
   {
      if (defaultTimeout == null)
         defaultTimeout = new BigInteger("300000");

      return defaultTimeout;
   }

   public void setDefaultTimeout(BigInteger value)
   {
      this.defaultTimeout = value;
   }

   @XmlAttribute
   public TAvailability getDefaultAvailability()
   {
      if (defaultAvailability == null)
         defaultAvailability = TAvailability.mandatory;

      return defaultAvailability;
   }

   public void setDefaultAvailability(TAvailability value)
   {
      this.defaultAvailability = value;
   }

   public Map<QName, String> getOtherAttributes()
   {
      return otherAttributes;
   }

   // Merge this Blueprint meta data with the given other, which may be null.
   public TBlueprint merge(TBlueprint other)
   {
      if (other == null)
         return this;

      throw new NotImplementedException("Blueprint metadata merge not yet implemented");
   }
}
