/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.osgi.framework.classloading;

import java.util.Map;
import java.util.StringTokenizer;

import org.jboss.classloading.plugins.metadata.PackageCapability;
import org.jboss.classloading.plugins.metadata.PackageRequirement;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classloading.spi.metadata.Requirement;
import org.jboss.classloading.spi.version.VersionRange;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.osgi.framework.bundle.OSGiBundleManager;
import org.jboss.osgi.framework.bundle.OSGiBundleState;
import org.jboss.osgi.framework.metadata.OSGiMetaData;
import org.jboss.osgi.framework.metadata.PackageAttribute;
import org.jboss.osgi.framework.metadata.Parameter;
import org.jboss.osgi.framework.metadata.ParameterizedAttribute;
import org.jboss.osgi.framework.metadata.internal.AbstractVersionRange;
import org.jboss.osgi.framework.resolver.BundleCapability;
import org.jboss.osgi.framework.resolver.BundleResolver;
import org.osgi.framework.Constants;
import org.osgi.framework.Version;

/**
 * OSGiPackageCapability.
 * 
 * todo PackagePermission/EXPORT todo uses todo include/exclude
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Thomas.Diesler@jboss.com
 * @version $Revision: 1.1 $
 */
public class OSGiPackageCapability extends PackageCapability
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 3940667616588052822L;

   /** The bundle state */
   private OSGiBundleState bundleState;

   /** The export package */
   private PackageAttribute exportPackage;

   /** The mandatory attributes */
   private String[] mandatoryAttributes;

   /**
    * Create a new OSGiPackageCapability.
    * 
    * @param metadata the osgi metadata
    * @param exportPackage the export package metadata
    * @return the capability
    * @throws IllegalArgumentException for null metadata
    */
   @SuppressWarnings("deprecation")
   public static OSGiPackageCapability create(OSGiBundleState bundleState, PackageAttribute exportPackage)
   {
      if (bundleState == null)
         throw new IllegalArgumentException("Null bundle");

      String name = exportPackage.getAttribute();
      String versionString = exportPackage.getParameterValue(Constants.VERSION_ATTRIBUTE, String.class);

      String oldVersionString = exportPackage.getParameterValue(Constants.PACKAGE_SPECIFICATION_VERSION, String.class);
      if (oldVersionString != null)
      {
         if (versionString != null && versionString.equals(oldVersionString) == false)
            throw new IllegalStateException(Constants.VERSION_ATTRIBUTE + " of " + versionString + " does not match " + Constants.PACKAGE_SPECIFICATION_VERSION
                  + " of " + oldVersionString);
         if (versionString == null)
            versionString = oldVersionString;
      }

      Version version = null;
      if (versionString != null)
      {
         // Handle version strings with quotes 
         if (versionString.startsWith("\"") && versionString.endsWith("\""))
            versionString = versionString.substring(1, versionString.length() - 1);

         version = Version.parseVersion(versionString);
      }

      OSGiPackageCapability capability = new OSGiPackageCapability(bundleState, name, version, exportPackage);
      capability.setSplitPackagePolicy(SplitPackagePolicy.First);

      return capability;
   }

   private OSGiPackageCapability(OSGiBundleState bundleState, String name, Version version, PackageAttribute exportPackage)
   {
      super(name, version);
      this.bundleState = bundleState;
      this.exportPackage = exportPackage;

      String mandatory = exportPackage.getParameterValue(Constants.MANDATORY_DIRECTIVE, String.class);
      if (mandatory != null)
      {
         StringTokenizer tokens = new StringTokenizer(mandatory, ",");
         mandatoryAttributes = new String[tokens.countTokens()];
         int i = 0;
         while (tokens.hasMoreTokens())
            mandatoryAttributes[i++] = tokens.nextToken();
      }

      if (exportPackage.getParameter(Constants.BUNDLE_SYMBOLICNAME_ATTRIBUTE) != null)
         throw new IllegalStateException("You cannot specify " + Constants.BUNDLE_SYMBOLICNAME_ATTRIBUTE + " on an Export-Package");
      if (exportPackage.getParameter(Constants.BUNDLE_VERSION_ATTRIBUTE) != null)
         throw new IllegalStateException("You cannot specify " + Constants.BUNDLE_VERSION_ATTRIBUTE + " on an Export-Package");
   }

   @Override
   public boolean resolves(Module reqModule, Requirement requirement)
   {
      if (super.resolves(reqModule, requirement) == false)
         return false;
      if (requirement instanceof OSGiPackageRequirement == false)
         return true;

      // Get the bundle resolver from the bundle manager
      OSGiBundleManager bundleManager = bundleState.getBundleManager();
      BundleResolver bundleResolver = bundleManager.getBundleResolver();
      
      // True if capModule matches with the resolved module
      PackageRequirement packageRequirement = (PackageRequirement)requirement;
      BundleCapability bundleCapability = bundleResolver.getMatchingCapability(reqModule, packageRequirement);
      if (bundleCapability == null)
         return false;

      Module capModule = getModule();
      Module otherCapModule = bundleCapability.getExportingModule();
      PackageCapability otherCapability = bundleCapability.getPackageCapability();
      boolean isMatch = (otherCapModule == capModule && otherCapability == this);
      return isMatch;
   }

   /**
    * Get the Module associated with this capability
    */
   public Module getModule()
   {
      DeploymentUnit unit = bundleState.getDeploymentUnit();
      Module module = unit.getAttachment(Module.class);
      if (module == null)
         throw new IllegalStateException("Cannot obtain module from: " + bundleState);
      return module;
   }

   @SuppressWarnings("deprecation")
   public boolean match(OSGiPackageRequirement packageRequirement)
   {
      String capPackageName = getName();
      String reqPackageName = packageRequirement.getName();
      if (capPackageName.equals(reqPackageName) == false)
         return false;
      
      VersionRange reqVersionRange = packageRequirement.getVersionRange();
      Object capVersion = getVersion();
      if (reqVersionRange.isInRange(capVersion) == false)
         return false;
      
      OSGiMetaData metaData = bundleState.getOSGiMetaData();
      PackageAttribute capParameters = exportPackage;
      PackageAttribute reqParameters = packageRequirement.getRequirePackage();
      
      boolean validMatch = true;

      // Check all the manadatory attributes are present
      if (validMatch == true && mandatoryAttributes != null)
      {
         if (reqParameters != null)
         {
            for (String mand : mandatoryAttributes)
            {
               if (reqParameters.getParameter(mand) == null)
               {
                  validMatch = false;
                  break;
               }
            }
         }
      }

      if (validMatch == true && reqParameters != null)
      {
         Map<String, Parameter> params = reqParameters.getParameters();
         if (params != null && params.isEmpty() == false)
         {
            for (String name : params.keySet())
            {
               String otherValue = reqParameters.getParameterValue(name, String.class);
               String ourValue = capParameters.getParameterValue(name, String.class);

               // todo we shouldn't mix attributes and directives in the metadata
               if (Constants.BUNDLE_SYMBOLICNAME_ATTRIBUTE.equals(name))
               {
                  if (otherValue.equals(metaData.getBundleSymbolicName()) == false)
                     validMatch = false;
               }
               else if (Constants.BUNDLE_VERSION_ATTRIBUTE.equals(name))
               {
                  VersionRange range = (VersionRange)AbstractVersionRange.valueOf(otherValue);
                  if (range.isInRange(metaData.getBundleVersion()) == false)
                     validMatch = false;
               }
               else if (Constants.RESOLUTION_DIRECTIVE.equals(name) == false && Constants.PACKAGE_SPECIFICATION_VERSION.equals(name) == false
                     && Constants.VERSION_ATTRIBUTE.equals(name) == false)
               {
                  if (ourValue == null || ourValue.equals(otherValue) == false)
                     validMatch = false;
               }

               if (validMatch == false)
                  break;
            }
         }
      }

      return validMatch;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof OSGiPackageCapability == false)
         return false;
      if (super.equals(obj) == false)
         return false;

      return true;
   }

   @Override
   protected void toString(StringBuffer buffer)
   {
      super.toString(buffer);
      OSGiMetaData metadata = bundleState.getOSGiMetaData();
      ParameterizedAttribute parameters = metadata.getBundleParameters();
      if (parameters != null)
      {
         Map<String, Parameter> params = parameters.getParameters();
         if (params != null && params.isEmpty() == false)
            buffer.append(" parameters=").append(params);
      }
   }
}
