/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

// This class is based on some original classes from
// Apache Felix which is licensed as below

/* 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.osgi.framework.filter.model;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Iterator;

import org.jboss.osgi.framework.filter.property.PropertySource;

/**
 * CompareOperation.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public abstract class CompareOperation extends Operation
{
   /** The left attribute */
   private String left;

   /** The right value */
   private String right;

   /**
    * Create a new CompareOperation.
    * 
    * @param left the attribute
    * @param right the value
    */
   public CompareOperation(String left, String right)
   {
      if (left == null)
         throw new IllegalArgumentException("Null left");
      if (right == null)
         throw new IllegalArgumentException("Null right");
      this.left = left;
      this.right = right;
   }

   @Override
   public boolean doMatch(PropertySource source)
   {
      Object value = source.getProperty(left);
      if (value == null)
         return false;
      return compare(value, right);
   }

   /**
    * Compare two values
    * 
    * @param lhs the left hand side
    * @param rhs the right hand side
    * @return true when they compare according to the operation
    */
   @SuppressWarnings("unchecked")
   protected boolean compare(Object lhs, String rhs)
   {
      if (lhs.getClass().isArray())
      {
         // TODO primitive arrays
         
         Object[] array = (Object[]) lhs;
         for (int i = 0; i < array.length; ++i)
         {
            if (compare(array[i], rhs))
               return true;
         }
         return false;
      }
      else if (lhs instanceof Iterable)
      {
         Iterable<?> iterable = Iterable.class.cast(lhs);
         Iterator<?> iterator = iterable.iterator();
         while (iterator.hasNext())
         {
            if (compare(iterator.next(), rhs))
               return true;
         }
         return false;
      }
      
      Object rhsValue = null;
      Class<?> lhsClass = lhs.getClass();
      try
      {
         // Convert to the correct type
         if (lhsClass == String.class)
            rhsValue = rhs;
         else if (lhsClass == Character.class)
            rhsValue = new Character(rhs.charAt(0)); // REVIEW Ignore remaining characters?
         else
         {
            // TODO use jboss-reflect
            try
            {
               Method method = lhsClass.getMethod("valueOf", new Class[] { String.class });
               rhsValue = method.invoke(null, rhs);
            }
            catch (Exception e)
            {
               Constructor<?> constructor = lhsClass.getConstructor(new Class[] { String.class });
               rhsValue = constructor.newInstance(rhs);
            }
         }
      }
      catch (Exception e)
      {
         throw new RuntimeException("Unable to create a comparable using new " + lhsClass.getName() + "(java.lang.String)", e);
      }
      
      if (Comparable.class.isAssignableFrom(lhs.getClass()) == false)
         return equals(lhs, rhsValue);

      Comparable<?> lhsComparable = Comparable.class.cast(lhs);
      return compare(lhsComparable, rhsValue);
   }

   /**
    * Compare using equals
    * 
    * @param lhs the attribute value
    * @param rhs the string value converted to the attribute's type
    * @return true when equal
    */
   protected boolean equals(Object lhs, Object rhs)
   {
      throw new IllegalArgumentException("Cannot do '" + toString() + " since " + lhs.getClass().getName() + " is not comparable");
   }

   /**
    * Compare using comparator
    * 
    * @param lhs the attribute value
    * @param rhs the string value converted to the attribute's type
    * @return true when equal
    */
   @SuppressWarnings("unchecked")
   protected abstract boolean compare(Comparable lhs, Object rhs);
   
   @Override
   public void toString(StringBuilder builder)
   {
      builder.append(left);
      appendOperation(builder);
      builder.append(right);
   }

   public abstract void appendOperation(StringBuilder builder);
}
