/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

// This class is based on some original classes from
// Apache Felix which is licensed as below

/* 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.osgi.framework.filter.model;

import java.util.List;

import org.jboss.osgi.framework.filter.property.PropertySource;

/**
 * SubStringOperation.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class SubStringOperation extends Operation
{
   /** The attribute */
   private String attribute;

   /** The prieces */
   private List<String> pieces;
   
   /**
    * Create a new SubStringOperation.
    * 
    * @param attribute the attribute
    * @param pieces the pieces
    */
   public SubStringOperation(String attribute, List<String> pieces)
   {
      if (attribute == null)
         throw new IllegalArgumentException("Null attribute");
      if (pieces == null)
         throw new IllegalStateException("Null pieces");
      this.attribute = attribute;
      this.pieces = pieces;
   }

   @Override
   public boolean doMatch(PropertySource source)
   {
      Object value = source.getProperty(attribute);
      if (value == null)
         return false;
      if (value instanceof String)
         return compare((String) value);
      else if (value instanceof String[])
      {
         String[] values = (String[]) value;
         for (String val : values)
         {
            if (compare(val))
               return true;
         }
         return false;
      }
      throw new RuntimeException("Attribute " + value + " is not a String or String[] type=" + value.getClass().getName());
   }

   /**
    * Compare the string to the pieces
    * 
    * @param value the string
    * @return true when it matchs
    */
   protected boolean compare(String value)
   {
      // Current index in value
      int index = 0;
      
      // Go through each piece
      int noPieces = pieces.size();
      for (int i = 0; i < noPieces; ++i)
      {
         String piece = pieces.get(i);
         
         // Last piece
         if (i == noPieces-1)
            return piece.length() <= value.length() - index && value.endsWith(piece);
         // First piece
         if (i == 0)
         {
            if (value.startsWith(piece) == false)
               return false;
         }
         // Middle pieces
         else
         {
            index = value.indexOf(piece, index);
            if (index == -1)
               return false;
         }
         index += piece.length();
      }
      return false;
   }
   
   @Override
   public void toString(StringBuilder builder)
   {
      builder.append(attribute).append('=');
      boolean first = true;
      for (String piece : pieces)
      {
         if (first)
            first = false;
         else
            builder.append('*');
         builder.append(piece);
      }
   }
}
