/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

// This class is based on some original classes from
// Apache Felix which is licensed as below

/* 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.osgi.framework.filter.parser;

import java.io.CharArrayReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import org.jboss.osgi.framework.filter.model.AndOperation;
import org.jboss.osgi.framework.filter.model.ApproximateOperation;
import org.jboss.osgi.framework.filter.model.EqualsOperation;
import org.jboss.osgi.framework.filter.model.GreaterThanOperation;
import org.jboss.osgi.framework.filter.model.LessThanOperation;
import org.jboss.osgi.framework.filter.model.NotOperation;
import org.jboss.osgi.framework.filter.model.Operation;
import org.jboss.osgi.framework.filter.model.OrOperation;
import org.jboss.osgi.framework.filter.model.PresentOperation;
import org.jboss.osgi.framework.filter.model.SubStringOperation;
import org.osgi.framework.InvalidSyntaxException;

/**
 * FilterParser.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class FilterParser
{
   /** EOF */
   private static final int EOF = -1;

   /** Left parenthesis */
   private static final int LEFT_PAREN = '(';

   /** Right parenthesis */
   private static final int RIGHT_PAREN = ')';

   /** Asterisk */
   private static final int ASTERISK = '*';

   /** Leading Attribute Characters */
   private static final String LEADING_ATTRIBUTE = ".abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-_";

   /** Internal Attribute Characters */
   private static final String INTERNAL_ATTRIBUTE = ".abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-_ ";

   /** Simple operation */
   private static final int SIMPLE = 0;

   /** Present operation */
   private static final int PRESENT = 1;

   /** Substring operation */
   private static final int SUBSTRING = 2;

   /** Next character not reader */
   private static final int NO_NEXT_CHAR = 0;
   
   /** The current character */
   private int nextChar = NO_NEXT_CHAR;
   
   /** The current position in the stream */
   private int position = 0;
   
   /** The filter */
   private String filter;
   
   /** The reader */
   private Reader reader;
   
   /**
    * Parse a filter from a string
    * 
    * @param filter the filter string
    * @return the operation
    * @throws InvalidSyntaxException for any error parsing the filter
    */
   public Operation parse(String filter) throws InvalidSyntaxException
   {
      if (filter == null)
         throw new InvalidSyntaxException("Null filter", "?");
      
      this.filter = filter;
      reader = new CharArrayReader(filter.toCharArray());

      try
      {
         Operation result = matchFilter();
         if (peekEOF() != EOF)
            throw raiseSyntaxError("Expected EOF but was '" + peekChar() + "'");
         return result;
      }
      catch (InvalidSyntaxException e)
      {
         throw e;
      }
      catch (Throwable t)
      {
         throw raiseSyntaxError("Error parsing filter", t);
      }
   }

   // filter = '(' filter-comp ')'
   private Operation matchFilter() throws Exception
   {
      if (peekNWS() != LEFT_PAREN)
         throw raiseSyntaxError("Expected ( but was '" + peekChar() + "'");
      eat();
      
      Operation result = matchFilterComp();

      if (peekNWS() != RIGHT_PAREN)
         throw raiseSyntaxError("Expected ) but was '" + peekChar() + "'");
      eat();
      
      return result;
   }

   // filter-comp = and | or | not | operation
   // and = '&' filter-list
   // or = '|' filter-list
   // not = '!' filter
   private Operation matchFilterComp() throws Exception
   {
      int c = peekNWS();
      switch (c)
      {
         case '&':
         case '|':
         {
            char op = nextChar();
            List<Operation> operations = matchFilterList();
            if (operations.isEmpty())
               throw raiseSyntaxError("Expected filters after " + op + "'");
            if (c == '&')
               return new AndOperation(operations);
            else
               return new OrOperation(operations);
         }
         case '!':
         {
            eat();
            Operation other = matchFilter();
            return new NotOperation(other);
         }
         default:
         {
            return matchOperation();
         }
      }
   }

   // filter-list = filter | filter filter-list
   private List<Operation> matchFilterList() throws Exception
   {
      List<Operation> operations = new ArrayList<Operation>();
      
      int c = peekNWS();
      while (c == LEFT_PAREN)
      {
         operations.add(matchFilter());
         c = peekNWS();
      }
      return operations;
   }
   
   // operation = simple | present | substring
   private Operation matchOperation() throws Exception
   {
      String attribute = matchAttribute();
      
      int op = peekNWS();
      char opChar = peekChar();
      switch (op)
      {
         case '=':
         {
            eat();
            break;
         }
         case '~':
         case '<':
         case '>':
         {
            eat();
            char next = peekChar();
            if (next != '=')
               throw raiseSyntaxError("Expected = but was '" + next + "'");
            eat();
            break;
         }
         default:
         {
            throw raiseSyntaxError("Expected an operation i,e. =|<=|>=|<= but was '" + opChar + "'");
         }
      }
      
      ArrayList<String> pieces = new ArrayList<String>();
      int kind = substring(pieces);
      if (op != '=' && kind != SIMPLE)
         throw raiseSyntaxError("Operation " + op + "= is only allowed in simple cases");
      
      switch (kind)
      {
         case SIMPLE:
         {
            switch (op)
            {
               case '=':
                  return new EqualsOperation(attribute, pieces.get(0));
               case '<':
                  return new LessThanOperation(attribute, pieces.get(0));
               case '>':
                  return new GreaterThanOperation(attribute, pieces.get(0));
               case '~':
                  return new ApproximateOperation(attribute, pieces.get(0));
            }
            throw new IllegalStateException("Unknown operation: " + opChar);
         }
         case PRESENT:
         {
            return new PresentOperation(attribute);
         }
         case SUBSTRING:
         {
            return new SubStringOperation(attribute, pieces);
         }
         default:
         {
            throw new IllegalStateException("Unknown kind=" + kind);
         }
      }
   }
   
   private String matchAttribute() throws Exception
   {
      int c = peekNWS();
      if (LEADING_ATTRIBUTE.indexOf(c) == -1)
         throw raiseSyntaxError(peekChar() + " is not allowed to start an attribute name");
      
      StringBuilder builder = new StringBuilder();
      do
      {
         builder.append(nextChar());
      }
      while (INTERNAL_ATTRIBUTE.indexOf(peek()) >= 0);
      
      // Trim trailing blanks
      int i = builder.length()-1;
      while (i > 0 && builder.charAt(i) == ' ')
         i--;
      builder.setLength(i+1);
      return builder.toString();
   }

   // substring = attr '=' initial any final
   // initial = () | value
   // any = '*' star-value
   // star-value = () | value '*' star-value
   // final = () | value
   int substring(ArrayList<String> pieces) throws Exception
   {
      StringBuilder builder = new StringBuilder();
      boolean wasStar = false; // last piece was a star
      boolean leftStar = false; // initial piece is a star
      boolean rightStar = false; // final piece is a star
      
      int c = NO_NEXT_CHAR;
      while (c != RIGHT_PAREN)
      {
         c = peek();
         switch (c)
         {
            case EOF:
            {
               throw raiseSyntaxError("Unexpected EOF");
            }
            case RIGHT_PAREN:
            {
               if (wasStar)
                  rightStar = true;
               else
                  pieces.add(builder.toString());
               break;
            }
            case '\\':
            {
               wasStar = false;
               eat();
               char nextChar = safeNextChar();
               builder.append(nextChar);
               break;
            }
            case ASTERISK:
            {
               if (wasStar)
                  throw raiseSyntaxError("Unexpected **");
               eat();
               if (builder.length() > 0)
                  pieces.add(builder.toString());
               builder.setLength(0);
               // Leading star
               if (pieces.isEmpty())
                  leftStar = true;
               wasStar = true;
               break;
            }
            default:
            {
               wasStar = false;
               builder.append(safeNextChar());
            }
         }
      }
      
      if (pieces.isEmpty())
         return PRESENT; // Check leftStar???
      
      if (leftStar || rightStar || pieces.size() > 1)
      {
         if (rightStar)
            pieces.add("");
         if (leftStar)
            pieces.add(0, "");
         return SUBSTRING;
      }
      
      return SIMPLE;
   }
   
   private InvalidSyntaxException raiseSyntaxError(String message) throws InvalidSyntaxException
   {
      throw new InvalidSyntaxException(message + " at position " + position + " filter='" + filter + "'", filter);
   }
   
   private InvalidSyntaxException raiseSyntaxError(String message, Throwable t) throws InvalidSyntaxException
   {
      throw new InvalidSyntaxException(message + " at position " + position + " filter='" + filter + "' (" + t.getClass().getName() + ": " + t.getMessage() + ")", filter, t);
   }
   
   private int read() throws Exception
   {
      ++position;
      return reader.read();
   }
   
   private void eat() throws Exception
   {
      next();
   }
   
   private char nextChar() throws Exception
   {
      return (char) next();
   }
   
   private char safeNextChar() throws Exception
   {
      char result = nextChar();
      if (result == EOF)
         throw raiseSyntaxError("Unexpected EOF");
      return result;
   }
   
   private int next() throws Exception
   {
      // We haven't read it yet
      if (nextChar == NO_NEXT_CHAR)
         return read();

      // We already peeked it
      int result = nextChar;
      nextChar = NO_NEXT_CHAR;
      return result;
   }
   
   private char peekChar() throws Exception
   {
      return (char) peek();
   }
   
   private int peek() throws Exception
   {
      // Haven't already peeked it
      if (nextChar == NO_NEXT_CHAR)
         nextChar = read();
      
      // Return the peeked value
      return nextChar;
   }
   
   private void skipWS() throws Exception
   {
      while (Character.isWhitespace(peekChar()))
         next();
   }
   
   private int peekNWS() throws Exception
   {
      skipWS();
      int result = peek();
      if (result == EOF)
         raiseSyntaxError("Unexpected EOF");
      return result;
   }
   
   private int peekEOF() throws Exception
   {
      skipWS();
      return peek();
   }
}
