/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.osgi;

import java.lang.reflect.Method;
import java.net.URL;
import java.util.Collection;
import java.util.Enumeration;

import junit.framework.AssertionFailedError;

import org.jboss.classloader.plugins.ClassLoaderUtils;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.deployers.plugins.main.MainDeployerImpl;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.osgi.framework.bundle.AbstractBundleState;
import org.jboss.osgi.framework.bundle.OSGiBundleManager;
import org.jboss.test.kernel.junit.MicrocontainerTestDelegate;
import org.jboss.virtual.AssembledDirectory;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;
import org.jboss.virtual.plugins.context.jar.JarUtils;
import org.jboss.virtual.plugins.vfs.helpers.SuffixesExcludeFilter;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;

/**
 * A OSGiTestDelegate
 * 
 * @author <a href="baileyje@gmail.com">John Bailey</a>
 * @version $Revision: 87848 $
 */
public class OSGiTestDelegate extends MicrocontainerTestDelegate
{
   private MainDeployerImpl mainDeployer;
   private OSGiBundleManager bundleManager;

   static
   {
      //FIXME the vfs is not initialising itself properly
      VFS.init();
      try
      {
         VFS.getRoot(new URL("file:/"));
      }
      catch (Exception ignored) {}
   }
   
   /**
    * Create a new OSGiTestDelegate.
    * 
    * @param clazz
    * @throws Exception
    */
   public OSGiTestDelegate(Class<?> clazz) throws Exception
   {
      super(clazz);
   }
   
   /**
    * Deploys the jboss-osgi-bootstrap.xml bean descriptor to setup OSGi deployers
    */
   protected void deploy() throws Exception
   {
      String common = "/META-INF/jboss-osgi-bootstrap.xml";
      URL url = getClass().getResource(common);
      if (url == null)
         throw new IllegalStateException(common + " not found");
      deploy(url);
      try
      {
         super.deploy();
         deployBundles();
      }
      catch (Throwable t)
      {
         undeploy();
         if (t instanceof Exception)
            throw (Exception) t;
         if (t instanceof Error)
            throw (Error) t;
         throw new RuntimeException("Error during deploy", t);
      }
   }
   
   protected void undeploy()
   {
      undeployBundles();
      super.undeploy();
   }

   protected void deployBundles() throws Exception
   {
      try
      {
         Method method = clazz.getMethod("deployBundles", new Class[] { OSGiTestDelegate.class });
         log.debug("Deploying Bundles...");
         method.invoke(null, this);
      }
      catch (NoSuchMethodException e)
      {
         log.debug("No deployBundles() in " + clazz.getName());
      }
   }

   protected void undeployBundles() 
   {
      OSGiBundleManager bundleManager = getBundleManager();
      Collection<AbstractBundleState> bundles = bundleManager.getBundles();
      for (AbstractBundleState bundleState : bundles)
      {
         try
         {
            if (bundleState.getBundleId() != 0)
               bundleManager.uninstall(bundleState);
         }
         catch (Throwable t)
         {
            getLog().warn("Error undeploying bundle: " + bundleState, t);
         }
      }
   }
   
   /**
    * Get the MainDeployer bean from the MC
    * 
    * @return MainDeployerImpl
    */
   protected MainDeployerImpl getMainDeployer()
   {
      if (mainDeployer == null)
         mainDeployer = getBean("MainDeployer", ControllerState.INSTALLED, MainDeployerImpl.class);
      return mainDeployer;
   }
   
   /**
    * Get the BundleManager bean from the MC
    * 
    * @return the BundleManager
    */
   protected OSGiBundleManager getBundleManager()
   {
      if (bundleManager == null)
      {
         bundleManager = getBean("OSGiBundleManager", ControllerState.INSTALLED, OSGiBundleManager.class);
         try
         {
            if (bundleManager.isActive() == false)
               bundleManager.startFramework();
         }
         catch (BundleException ex)
         {
            throw new IllegalStateException("Cannot start bundle manager", ex);
         }
      }
      return bundleManager;
   }

   /**
    * Create a bundle 
    * 
    * @param root the location of the location to deploy
    * @param child the child to deploy
    * @return Bundle for the deployment
    * @throws Exception for any error
    */
   public Bundle addBundle(String root, String child) throws Exception
   {
      URL resourceRoot = getClass().getResource(root);
      if (resourceRoot == null)
         throw new AssertionFailedError("Resource not found: " + root);
      URL childResource = new URL(resourceRoot, child);
      AbstractBundleState bundleState = getBundleManager().install(childResource);
      return bundleState.getBundleInternal();
   }

   /**
    * Create a bundle 
    *
    * @param file the virtual file
    * @return Bundle for the deployment
    * @throws Exception for any error
    */
   public Bundle addBundle(VirtualFile file) throws Exception
   {
      AbstractBundleState bundleState = getBundleManager().install(file);
      return bundleState.getBundleInternal();
   }

   /**
    * Remove a bundle 
    * 
    * @param bundle the bundle to remove
    * @throws Exception for any error
    */
   public void uninstall(Bundle bundle) throws Exception
   {
      if (bundle.getState() != Bundle.UNINSTALLED)
         getBundleManager().uninstall(bundle);
   }
   
   protected DeploymentUnit getDeploymentUnit(Bundle bundle)
   {
      DeploymentUnit deploymentUnit = getBundleManager().getDeployment(bundle.getBundleId());
      if (deploymentUnit == null)
         throw new AssertionFailedError("Bundle not installed: " + bundle);
      return deploymentUnit;
   }

   public Bundle assembleBundle(String name, String resources, Class<?>... packages) throws Exception
   {
      return assembleBundle(name, new String[] { resources }, packages);
   }

   public Bundle assembleBundle(String name, String[] resourcePaths, Class<?>... packages) throws Exception
   {
      AssembledDirectory assembledDirectory = createAssembledDirectory(name, "");
      for (String path : resourcePaths)
         addPath(assembledDirectory, path, "");
      for (Class<?> reference : packages)
         addPackage(assembledDirectory, reference);
      return addBundle(assembledDirectory);
   }

   public AssembledDirectory createAssembledDirectory(String name) throws Exception
   {
      return createAssembledDirectory(name, "");     
   }

   public AssembledDirectory createAssembledDirectory(String name, String rootName) throws Exception
   {
      return AssembledDirectory.createAssembledDirectory(name, rootName);
   }

   public void addPackage(AssembledDirectory dir, Class<?> reference) throws Exception
   {
      String packagePath = ClassLoaderUtils.packageNameToPath(reference.getName());
      dir.addResources(reference, new String[] { packagePath + "/*.class" } , new String[0]);
   }

   public void addPath(final AssembledDirectory dir, String path, String name) throws Exception
   {
      URL url = getClass().getResource(path);
      if (url == null)
         throw new AssertionFailedError(path + " not found");

      VirtualFile file = VFS.getVirtualFile(url, name);
      // TODO - remove this filter after new VFS relase
      SuffixesExcludeFilter noJars = new SuffixesExcludeFilter(JarUtils.getSuffixes());
      dir.addPath(file, noJars);
   }
   
   protected URL getBundleResource(Bundle bundle, String path)
   {
      return getDeploymentUnit(bundle).getResourceLoader().getResource(path);
   }
   
   protected Enumeration<URL> getBundleResources(Bundle bundle, String path) throws Exception
   {
      return getDeploymentUnit(bundle).getResourceLoader().getResources(path);
   }
}
