/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.osgi.bundle.test;

import java.net.URL;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;

import junit.framework.Test;

import org.jboss.test.osgi.OSGiTestCase;
import org.osgi.framework.Bundle;

/**
 * BundleEntriesUnitTestCase.
 * 
 * TODO test security
 * TODO test fragments
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class BundleEntriesUnitTestCase extends OSGiTestCase
{
   public BundleEntriesUnitTestCase(String name)
   {
      super(name);
   }

   public void testEntriesNotInstalled() throws Exception
   {
      Bundle bundle = addBundle("/bundles/entries/", "entries-simple");
      uninstall(bundle);
      try
      {
         bundle.getEntry("root.xml");
         fail("Should not be here!");
      }
      catch (Throwable t)
      {
         checkThrowable(IllegalStateException.class, t);
      }
      try
      {
         bundle.findEntries("", "root.xml", false);
         fail("Should not be here!");
      }
      catch (Throwable t)
      {
         checkThrowable(IllegalStateException.class, t);
      }
   }
   
   public void testFindEntriesNoPath() throws Exception
   {
      Bundle bundle = addBundle("/bundles/entries/", "entries-simple");
      try
      {
         bundle.findEntries(null, "root.xml", false);
         fail("Should not be here!");
      }
      catch (Throwable t)
      {
         checkThrowable(IllegalArgumentException.class, t);
      }
      finally
      {
         uninstall(bundle);
      }
   }
   
   public void testEntries() throws Exception
   {
      Bundle bundle = addBundle("/bundles/entries/", "entries-simple");
      try
      {
         assertEntry(bundle, "");
         assertNoEntries(bundle, "", "", false);
         assertNoEntries(bundle, "", "", true);
         assertEntryPaths("", bundle, 
               "/",
               "root.xml",
               "root-no-suffix",
               "entry1.xml",
               "META-INF/",
               "META-INF/MANIFEST.MF",
               "org/",
               "org/jboss/",
               "org/jboss/test/",
               "org/jboss/test/osgi/",
               "org/jboss/test/osgi/bundle/",
               "org/jboss/test/osgi/bundle/entries/",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertNoEntry(bundle, "DoesNotExist");
         assertNoEntries(bundle, "", "DoesNotExist", false);
         assertNoEntries(bundle, "", "DoesNotExist", true);
         assertNoEntryPaths(bundle, "DoesNotExist");

         assertEntry(bundle, "root-no-suffix");
         assertEntries(bundle, "", "root-no-suffix", false, "root-no-suffix");
         assertEntries(bundle, "", "root-no-suffix", true, "root-no-suffix");
         assertEntryPaths(bundle, "root-no-suffix");

         assertEntry(bundle, "root.xml");
         assertEntries(bundle, "", "root.xml", false, "root.xml");
         assertEntries(bundle, "", "root.xml", true, "root.xml");
         assertEntryPaths(bundle, "root.xml");

         assertEntry(bundle, "entry1.xml");
         assertEntries(bundle, "", "entry1.xml", false, "entry1.xml");
         assertEntries(bundle, "", "entry1.xml", true, 
               "entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml"
         );
         assertEntryPaths(bundle, "entry1.xml");

         assertEntry(bundle, "META-INF");
         assertNoEntries(bundle, "", "META-INF", false);
         assertNoEntries(bundle, "", "META-INF", true);
         assertEntryPaths("META-INF", bundle, 
               "META-INF/",
               "META-INF/MANIFEST.MF"
         );
         
         assertNoEntry(bundle, "META-INF/DoesNotExist");
         assertNoEntries(bundle, "META-INF", "DoesNotExist", false);
         assertNoEntries(bundle, "META-INF", "DoesNotExist", true);
         assertNoEntryPaths(bundle, "META-INF/DoesNotExist");

         assertEntry(bundle, "META-INF/MANIFEST.MF");
         assertEntries(bundle, "META-INF", "MANIFEST.MF", false, "META-INF/MANIFEST.MF");
         assertEntries(bundle, "META-INF", "MANIFEST.MF", true, "META-INF/MANIFEST.MF");
         assertEntryPaths(bundle, "META-INF/MANIFEST.MF");

         assertEntry(bundle, "org");
         assertNoEntries(bundle, "", "org", false);
         assertNoEntries(bundle, "", "org", true);
         assertEntryPaths("org", bundle, 
               "org/",
               "org/jboss/",
               "org/jboss/test/",
               "org/jboss/test/osgi/",
               "org/jboss/test/osgi/bundle/",
               "org/jboss/test/osgi/bundle/entries/",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertNoEntry(bundle, "org/DoesNotExist");
         assertNoEntries(bundle, "org", "DoesNotExist", false);
         assertNoEntries(bundle, "org", "DoesNotExist", true);
         assertNoEntryPaths(bundle, "org/DoesNotExist");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries");
         assertEntries(bundle, "", "org/jboss/test/osgi/bundle/entries", false);
         assertEntries(bundle, "", "org/jboss/test/osgi/bundle/entries", true);
         assertEntryPaths("org/jboss/test/osgi/bundle/entries", bundle, 
               "org/jboss/test/osgi/bundle/entries/",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertNoEntry(bundle, "org/jboss/test/osgi/bundle/DoesNotExist");
         assertNoEntries(bundle, "org/jboss/test/osgi/bundle", "DoesNotExist", false);
         assertNoEntries(bundle, "org/jboss/test/osgi/bundle", "DoesNotExist", true);
         assertNoEntryPaths(bundle, "org/jboss/test/osgi/bundle/DoesNotExist");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/notxml.suffix");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "notxml.suffix", false, 
               "org/jboss/test/osgi/bundle/entries/notxml.suffix");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "notxml.suffix", true, 
               "org/jboss/test/osgi/bundle/entries/notxml.suffix"
         );
         assertEntryPaths(bundle, "org/jboss/test/osgi/bundle/entries/notxml.suffix");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/entry1.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry1.xml", false, 
               "org/jboss/test/osgi/bundle/entries/entry1.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry1.xml", true, 
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml"
         );
         assertEntryPaths(bundle, "org/jboss/test/osgi/bundle/entries/entry1.xml");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/entry2.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry2.xml", false, 
               "org/jboss/test/osgi/bundle/entries/entry2.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry2.xml", true, 
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         assertEntryPaths(bundle, "org/jboss/test/osgi/bundle/entries/entry2.xml");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/sub");
         assertEntries(bundle, "", "org/jboss/test/osgi/bundle/entries/sub", false);
         assertEntries(bundle, "", "org/jboss/test/osgi/bundle/entries/sub", true);
         assertEntryPaths("org/jboss/test/osgi/bundle/entries/sub", bundle, 
               "org/jboss/test/osgi/bundle/entries/sub/",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertNoEntry(bundle, "org/jboss/test/osgi/bundle/DoesNotExist/sub");
         assertNoEntries(bundle, "org/jboss/test/osgi/bundle/sub", "DoesNotExist", false);
         assertNoEntries(bundle, "org/jboss/test/osgi/bundle/sub", "DoesNotExist", true);
         assertNoEntryPaths(bundle, "org/jboss/test/osgi/bundle/DoesNotExist/sub");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/sub/entry1.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries/sub", "entry1.xml", false, 
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries/sub", "entry1.xml", true, 
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml");
         assertEntryPaths(bundle, "org/jboss/test/osgi/bundle/entries/sub/entry1.xml");

         assertEntry(bundle, "org/jboss/test/osgi/bundle/entries/sub/entry2.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries/sub", "entry2.xml", false, 
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml");
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries/sub", "entry2.xml", true, 
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml");
         assertEntryPaths(bundle, "org/jboss/test/osgi/bundle/entries/sub/entry2.xml");

         assertEntries(bundle, "", "*", false,
               "root.xml",
               "root-no-suffix",
               "entry1.xml"
         );
         assertEntries(bundle, "", "*", true,
               "root.xml",
               "root-no-suffix",
               "entry1.xml",
               "META-INF/MANIFEST.MF",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );

         assertEntries(bundle, "", null, false,
               "root.xml",
               "root-no-suffix",
               "entry1.xml"
         );
         assertEntries(bundle, "", null, true,
               "root.xml",
               "root-no-suffix",
               "entry1.xml",
               "META-INF/MANIFEST.MF",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertEntries(bundle, "", "root*", false,
               "root-no-suffix",
               "root.xml"
         );
         assertEntries(bundle, "", "root*", true,
               "root-no-suffix",
               "root.xml"
         );
         
         assertEntries(bundle, "", "entry*", false,
               "entry1.xml"
         );
         assertEntries(bundle, "", "entry*", true,
               "entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry*", false,
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml"
         );
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "entry*", true,
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertEntries(bundle, "", "*.xml", false,
               "root.xml",
               "entry1.xml"
         );
         assertEntries(bundle, "", "*.xml", true,
               "root.xml",
               "entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertEntries(bundle, "", "*xml*", false,
               "root.xml",
               "entry1.xml"
         );
         assertEntries(bundle, "", "*xml*", true,
               "root.xml",
               "entry1.xml",
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
         
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "*xml*", false,
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml"
         );
         assertEntries(bundle, "org/jboss/test/osgi/bundle/entries", "*xml*", true,
               "org/jboss/test/osgi/bundle/entries/notxml.suffix",
               "org/jboss/test/osgi/bundle/entries/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry1.xml",
               "org/jboss/test/osgi/bundle/entries/entry2.xml",
               "org/jboss/test/osgi/bundle/entries/sub/entry2.xml"
         );
      }
      finally
      {
         uninstall(bundle);
      }
   }

   protected void assertEntry(Bundle bundle, String path) throws Exception
   {
      URL expected = getBundleResource(bundle, path);
      if (expected == null)
         fail("Expected resource not found: " + path);

      assertEntry(bundle, path, expected);
      assertEntry(bundle, "/" + path, expected);
   }

   protected void assertEntry(Bundle bundle, String path, URL expected) throws Exception
   {
      URL actual = bundle.getEntry(path);
      assertEquals(expected, actual);
   }

   protected void assertNoEntry(Bundle bundle, String path) throws Exception
   {
      URL actual = bundle.getEntry(path);
      assertNull("Did not expect entry: " + actual + " for path: " + path, actual);
   }
   
   protected void assertEntries(Bundle bundle, String path, String filePattern, boolean recurse, String... entries) throws Exception
   {
      Set<URL> expected = new HashSet<URL>();
      for (String entry : entries)
      {
         Enumeration<URL> urls = getBundleResources(bundle, entry);
         if (urls == null || urls.hasMoreElements() == false)
            fail("Expected resource not found: " + entry);
         while (urls.hasMoreElements())
            expected.add(urls.nextElement());
      }

      assertEntries(bundle, path, filePattern, recurse, expected);
      assertEntries(bundle, "/" + path, filePattern, recurse, expected);
   }
   
   @SuppressWarnings("unchecked")
   protected void assertEntries(Bundle bundle, String path, String filePattern, boolean recurse, Set<URL> expected) throws Exception
   {
      Set<URL> actual = new HashSet<URL>();
      Enumeration<URL> enumeration = bundle.findEntries(path, filePattern, recurse);
      while (enumeration != null && enumeration.hasMoreElements())
         actual.add(enumeration.nextElement());
      
      assertEquals(expected, actual);
   }

   protected void assertNoEntries(Bundle bundle, String path, String filePattern, boolean recurse) throws Exception
   {
      assertEntries(bundle, path, filePattern, recurse);
   }
   
   protected void assertEntryPaths(Bundle bundle, String path) throws Exception
   {
      Set<String> expected = Collections.singleton(path);

      assertEntryPaths(bundle, path, expected);
      assertEntryPaths(bundle, "/" + path, expected);
   }
   
   protected void assertEntryPaths(String path, Bundle bundle, String... entries) throws Exception
   {
      Set<String> expected = new HashSet<String>();
      expected.addAll(Arrays.asList(entries));

      assertEntryPaths(bundle, path, expected);
      assertEntryPaths(bundle, "/" + path, expected);
   }
   
   @SuppressWarnings("unchecked")
   protected void assertEntryPaths(Bundle bundle, String path, Set<String> expected) throws Exception
   {
      Set<String> actual = new HashSet<String>();
      Enumeration<String> enumeration = bundle.getEntryPaths(path);
      while (enumeration != null && enumeration.hasMoreElements())
         actual.add(enumeration.nextElement());
      
      assertEquals(expected, actual);
   }
   
   @SuppressWarnings("unchecked")
   protected void assertNoEntryPaths(Bundle bundle, String path) throws Exception
   {
      Enumeration<String> enumeration = bundle.getEntryPaths(path);
      if (enumeration != null)
      {
         Set<String> actual = new HashSet<String>();
         while (enumeration.hasMoreElements())
            actual.add(enumeration.nextElement());
         fail("For path " + path + " did not expect entry paths: " + actual);
      }
   }
}
