/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.osgi.classloader.test;

import org.jboss.test.osgi.OSGiTestCase;
import org.jboss.test.osgi.classloader.support.a.A;
import org.jboss.test.osgi.classloader.support.b.B;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;

/**
 * ExportImportUnitTestCase.
 * 
 * TODO test security 
 * TODO test mandatory attributes 
 * TODO test include/exclude 
 * TODO test uses
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Thomas.Diesler@jboss.com
 * @version $Revision: 1.1 $
 */
public class ExportImportPackageUnitTestCase extends OSGiTestCase
{
   public ExportImportPackageUnitTestCase(String name)
   {
      super(name);
   }

   public void testSimpleImportPackage() throws Exception
   {
      //Bundle-Name: BundleA
      //Bundle-Version: 1.0.0
      //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleA;test=x
      //Export-Package: org.jboss.test.osgi.classloader.support.a;version=1.0.0;test=x
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         
         //Bundle-Name: BundleB
         //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleB
         //Export-Package: org.jboss.test.osgi.classloader.support.b
         //Import-Package: org.jboss.test.osgi.classloader.support.a
         Bundle bundle2 = assembleBundle("simpleimportpackageA", "/bundles/classloader/simpleimportpackageA", B.class);
         
         try
         {
            bundle2.start();
            assertLoadClass(bundle2, A.class, bundle1);
            assertLoadClass(bundle2, B.class, bundle2);

            assertLoadClassFail(bundle1, B.class);
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testSimpleImportPackageFails() throws Exception
   {
      //Bundle-Name: BundleA
      //Bundle-Version: 1.0.0
      //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleA;test=x
      //Export-Package: org.jboss.test.osgi.classloader.support.a;version=1.0.0;test=x
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         
         //Bundle-Name: BundleB
         //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleB
         //Import-Package: doesnotexist
         Bundle bundle2 = assembleBundle("simpleimportpackagefails", "/bundles/classloader/simpleimportpackagefails", B.class);
         
         try
         {
            bundle2.start();
            fail("Should not be here!");
         }
         catch (BundleException ex)
         {
            // expected
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testSelfImportPackage() throws Exception
   {
      //Bundle-Name: BundleB
      //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleB
      //Export-Package: org.jboss.test.osgi.classloader.support.b
      //Import-Package: org.jboss.test.osgi.classloader.support.a
      Bundle bundleB = assembleBundle("BundleB", "/bundles/classloader/simpleimportpackageA", B.class);
      
      try
      {
         assertEquals("Bundle INSTALLED", Bundle.INSTALLED, bundleB.getState());
         
         try
         {
            bundleB.start();
            fail("Expected to fail due to unresolved import");
         }
         catch (BundleException ex)
         {
            // expected
         }
         
         //Bundle-Name: BundleA
         //Bundle-Version: 1.0.0
         //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleA
         //Export-Package: org.jboss.test.osgi.classloader.support.a, org.jboss.test.osgi.classloader.support.b 
         //Import-Package: org.jboss.test.osgi.classloader.support.a, org.jboss.test.osgi.classloader.support.b 
         Bundle bundleA = assembleBundle("BundleA", "/bundles/classloader/selfimportpackageAB", A.class, B.class);
         
         try
         {
            bundleB.start();
            assertEquals("Bundle ACTIVE", Bundle.ACTIVE, bundleB.getState());
            
            // BundleA is expected to resolve when BundleB gets started
            assertEquals("Bundle RESOLVED", Bundle.RESOLVED, bundleA.getState());
            assertLoadClass(bundleA, A.class, bundleA);
            assertLoadClass(bundleA, B.class, bundleA);

            // BundleB imports A from BundleA
            assertLoadClass(bundleB, A.class, bundleA);
            
            // BundleB does not import B
            assertLoadClass(bundleB, B.class, bundleB);
         }
         finally
         {
            uninstall(bundleA);
         }
      }
      finally
      {
         uninstall(bundleB);
      }
   }

   public void testVersionImportPackage() throws Exception
   {
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         Bundle bundle2 = assembleBundle("versionimportpackageA", "/bundles/classloader/versionimportpackageA", B.class);
         try
         {
            bundle2.start();
            assertLoadClass(bundle2, A.class, bundle1);
            assertLoadClass(bundle2, B.class, bundle2);
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testVersionImportPackageFails() throws Exception
   {
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         Bundle bundle2 = assembleBundle("versionimportpackagefails", "/bundles/classloader/versionimportpackagefails", B.class);
         try
         {
            bundle2.start();
            fail("Should not be here!");
         }
         catch (BundleException ex)
         {
            // expected
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testOptionalImportPackage() throws Exception
   {
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         Bundle bundle2 = assembleBundle("optionalimportpackageA", "/bundles/classloader/optionalimportpackageA", B.class);
         try
         {
            bundle2.start();
            assertLoadClass(bundle2, A.class, bundle1);
            assertLoadClass(bundle2, B.class, bundle2);
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testOptionalImportPackageFails() throws Exception
   {
      Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
      try
      {
         bundle1.start();
         assertLoadClass(bundle1, A.class);
         Bundle bundle2 = assembleBundle("optionalimportpackagefails", "/bundles/classloader/optionalimportpackagefails", B.class);
         try
         {
            bundle2.start();
            assertLoadClassFail(bundle2, A.class);
            assertLoadClass(bundle2, B.class);
         }
         finally
         {
            uninstall(bundle2);
         }
      }
      finally
      {
         uninstall(bundle1);
      }
   }

   public void testBundleNameImportPackage() throws Exception
   {
      //Bundle-Name: NotBundleA
      //Bundle-Version: 1.0.0
      //Bundle-SymbolicName: org.jboss.test.osgi.classloader.notbundleA
      //Export-Package: org.jboss.test.osgi.classloader.support.a
      Bundle bundle0 = assembleBundle("notbundleA", "/bundles/classloader/notbundleA", A.class);
      
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         assertLoadClass(bundle0, A.class, bundle0);
         
         //Bundle-Name: BundleA
         //Bundle-Version: 1.0.0
         //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleA;test=x
         //Export-Package: org.jboss.test.osgi.classloader.support.a;version=1.0.0;test=x
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            assertLoadClass(bundle1, A.class, bundle1);
            
            //Bundle-Name: BundleB
            //Bundle-SymbolicName: org.jboss.test.osgi.classloader.bundleB
            //Import-Package: org.jboss.test.osgi.classloader.support.a;bundle-symbolic-name=org.jboss.test.osgi.classloader.bundleA
            Bundle bundle2 = assembleBundle("bundlenameimportpackageA", "/bundles/classloader/bundlenameimportpackageA", B.class);
            
            try
            {
               bundle2.start();
               System.out.println("FIXME [JBOSGI-153] Cannot parse Import-Package parameters");
               //assertLoadClass(bundle2, A.class, bundle1);
               
               assertLoadClass(bundle2, B.class, bundle2);
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }

   public void testBundleNameImportPackageFails() throws Exception
   {
      Bundle bundle0 = assembleBundle("notbundleA", "/bundles/classloader/notbundleA", A.class);
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            Bundle bundle2 = assembleBundle("bundlenameimportpackagefails", "/bundles/classloader/bundlenameimportpackagefails", B.class);
            try
            {
               bundle2.start();
               fail("Should not be here!");
            }
            catch (BundleException ex)
            {
               // expected
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }

   public void testBundleVersionImportPackage() throws Exception
   {
      Bundle bundle0 = assembleBundle("bundleA2", "/bundles/classloader/bundleA2", A.class);
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            Bundle bundle2 = assembleBundle("bundleversionimportpackageA", "/bundles/classloader/bundleversionimportpackageA", B.class);
            try
            {
               bundle2.start();
               assertLoadClass(bundle2, A.class, bundle1);
               assertLoadClass(bundle2, B.class, bundle2);
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }

   public void testBundleVersionImportPackageFails() throws Exception
   {
      Bundle bundle0 = assembleBundle("bundleA2", "/bundles/classloader/bundleA2", A.class);
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            Bundle bundle2 = assembleBundle("bundleversionimportpackagefails", "/bundles/classloader/bundleversionimportpackagefails", B.class);
            try
            {
               bundle2.start();
               fail("Should not be here!");
            }
            catch (BundleException ex)
            {
               // expected
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }

   public void testAttributeImportPackage() throws Exception
   {
      Bundle bundle0 = assembleBundle("bundleA2", "/bundles/classloader/bundleA2", A.class);
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            Bundle bundle2 = assembleBundle("attributeimportpackageA", "/bundles/classloader/attributeimportpackageA", B.class);
            try
            {
               bundle2.start();
               assertLoadClass(bundle2, A.class, bundle1);
               assertLoadClass(bundle2, B.class, bundle2);
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }

   public void testAttributeImportPackageFails() throws Exception
   {
      Bundle bundle0 = assembleBundle("bundleA2", "/bundles/classloader/bundleA2", A.class);
      try
      {
         bundle0.start();
         assertLoadClass(bundle0, A.class);
         Bundle bundle1 = assembleBundle("bundleA", "/bundles/classloader/bundleA", A.class);
         try
         {
            bundle1.start();
            assertLoadClass(bundle1, A.class);
            Bundle bundle2 = assembleBundle("attributeimportpackagefails", "/bundles/classloader/attributeimportpackagefails", B.class);
            try
            {
               bundle2.start();
               fail("Should not be here!");
            }
            catch (BundleException ex)
            {
               // expected
            }
            finally
            {
               uninstall(bundle2);
            }
         }
         finally
         {
            uninstall(bundle1);
         }
      }
      finally
      {
         uninstall(bundle0);
      }
   }
}
