/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.test.osgi.filter.test;

import java.util.Dictionary;
import java.util.Hashtable;

import org.jboss.test.osgi.OSGiTestCase;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;

/**
 * AbstractFilterTest.
 * 
 * TODO test type conversions
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public abstract class AbstractFilterTest extends OSGiTestCase
{
   public AbstractFilterTest(String name)
   {
      super(name);
   }

   protected abstract Filter createFilter(String filter) throws Exception;

   public void testBasic() throws Exception
   {
      // Null filter
      assertInvalid(null);
      // Empty
      assertInvalid("");
      // No left parenthesis
      assertInvalid("a=b)");
      // No right parenthesis
      assertInvalid("(a=b");
      // No filter
      assertInvalid("()");
      // Missing EOF
      assertInvalid("(a=b)A");

      // Simple filter
      assertValid("(a=b)");
      // Simple Whitespace tests
      assertValid(" (a=b)");
      assertValid("(a=b) ");
   }
   
   public void testEquals() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b")
      .put("c", " d")
      .put("e", "f ")
      .put("star", "*")
      .put("slash", "\\")
      .put("rightparen", ")");

      assertInvalid("(=)");
      assertInvalid("(=");
      assertInvalid("(=b)");
      assertInvalid("(!=b)");
      assertInvalid("(&=b)");
      assertInvalid("(|=b)");
      assertInvalid("(<=b)");
      assertInvalid("(>=b)");
      assertInvalid("(~=b)");
      System.out.println("FIXME [JBOSGI-129] - Invalid (*=b)");
      assertNoMatch("(*=b)", properties);
      assertInvalid("(a=\\)");
      
      assertMatch("(a=b)", properties);
      assertMatch("( a=b)", properties);
      assertMatch("(a =b)", properties);
      assertMatch("(c= d)", properties);
      assertMatch("(e=f )", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a=)");
      assertInvalid("(a=)");
      assertNoMatch("(a=c)", properties);
      assertNoMatch("(x=c)", properties);
      
      assertMatch("(star=\\*)", properties);
      assertNoMatch("(a=\\*)", properties);
      assertMatch("(slash=\\\\)", properties);
      assertNoMatch("(a=\\\\)", properties);
      assertMatch("(rightparen=\\))", properties);
      assertNoMatch("(a=\\))", properties);
   }
   
   public void testLess() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", 1)
      .put("string", "2");

      assertInvalid("(<)");
      assertInvalid("(< =)");
      assertInvalid("(<=)");
      assertInvalid("(<=");
      assertInvalid("(<=b)");
      assertInvalid("(a<=)");
      
      assertMatch("(a<=2)", properties);
      assertMatch("( a<=2)", properties);
      assertMatch("(a <=2)", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a<= 2)");
      assertMatch("(a<= 2)", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a<=2 )");
      assertMatch("(a<=2 )", properties);
      assertNoMatch("(a<=0)", properties);
      assertNoMatch("( a<=0)", properties);
      assertNoMatch("(a <=0)", properties);
      assertNoMatch("(a<= 0)", properties);
      assertNoMatch("(a<=0 )", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a<= )");
      try
      {
         assertNoMatch("(a<= )", properties);
         fail("NumberFormatException expected");
      }
      catch (NumberFormatException e)
      {
         // expected
      }

      assertMatch("(string<=3)", properties);
      assertMatch("(string<=3 )", properties);
      assertNoMatch("(string<=1)", properties);
      assertNoMatch("(string<= 3)", properties);
   }
   
   public void testGreater() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", 1)
      .put("string", "2");

      assertInvalid("(>)");
      assertInvalid("(> =)");
      assertInvalid("(>=)");
      assertInvalid("(>=");
      assertInvalid("(>=b)");
      assertInvalid("(a>=)");
      
      assertMatch("(a>=0)", properties);
      assertMatch("( a>=0)", properties);
      assertMatch("(a >=0)", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a>= 0)");
      assertMatch("(a>= 0)", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a>=0 )");
      assertMatch("(a>=0 )", properties);
      assertNoMatch("(a>=2)", properties);
      assertNoMatch("( a>=2)", properties);
      assertNoMatch("(a >=2)", properties);
      assertNoMatch("(a>= 2)", properties);
      assertNoMatch("(a>=2 )", properties);
      System.out.println("FIXME [JBOSGI-129] - NoMatch (a>= )");
      try
      {
         assertNoMatch("(a>= )", properties);
         fail("NumberFormatException expected");
      }
      catch (NumberFormatException e)
      {
         // expected
      }

      assertMatch("(string>=1)", properties);
      assertMatch("(string>=1 )", properties);
      assertNoMatch("(string>=3)", properties);
      assertMatch("(string>= 1)", properties);
   }
   
   public void testApprox() throws Exception
   {
      assertInvalid("(~)");
      assertInvalid("(~ =)");
      assertInvalid("(~=)");
      assertInvalid("(~=");
      assertInvalid("(~=b)");
      
      // TODO testApprox
   }
   
   public void testPresent() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b");

      assertInvalid("(=*)");
      assertInvalid("(= *)");
      
      assertMatch("(a=*)", properties);
      assertMatch("( a=*)", properties);
      assertMatch("(a =*)", properties);
      assertNoMatch("(x=*)", properties);
      assertNoMatch("( x=*)", properties);
      assertNoMatch("(x =*)", properties);
   }
   
   public void testSubstring() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "abcdef")
      .put("astar", "a*")
      .put("aslash", "a\\")
      .put("arightparen", "a)");

      assertInvalid("(=**)");
      assertInvalid("(=a**b)");
      
      assertMatch("(a=*bcdef)", properties);
      assertMatch("( a=*bcdef)", properties);
      assertMatch("(a =*bcdef)", properties);
      assertMatch("(a=abcde*)", properties);
      assertMatch("(a=ab*def)", properties);
      assertMatch("(a=a*def)", properties);
      assertMatch("(a=a*ef)", properties);
      assertMatch("(a=a*f)", properties);
      assertMatch("(a=a*)", properties);
      assertMatch("(a=*f)", properties);
      assertMatch("(a=*bcde*)", properties);
      assertMatch("(a=*b*de*)", properties);
      assertNoMatch("(x=*bcdef)", properties);
      assertNoMatch("( x=*bcdef)", properties);
      assertNoMatch("(x =*bcdef)", properties);
      assertNoMatch("(a = *bcdef)", properties);
      assertNoMatch("(a =*bcdef )", properties);
      assertNoMatch("(a=x*)", properties);
      assertNoMatch("(a=*x)", properties);
      assertNoMatch("(a=xbcde*)", properties);
      assertNoMatch("(a=*bcdex)", properties);
      assertNoMatch("(a=*bcxe*)", properties);
      assertNoMatch("(a=*b*xe*)", properties);
      assertMatch("(a=ab*d*ef)", properties);
      assertNoMatch("(a=ab*d*def)", properties);
      
      assertMatch("(astar=*\\*)", properties);
      assertMatch("(aslash=*\\\\)", properties);
      assertMatch("(arightparen=*\\))", properties);
   }
   
   public void testNot() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b")
      .put("c", " d")
      .put("e", "f ");

      assertInvalid("(!)");
      assertInvalid("(!())");
      assertInvalid("(!(x=))");
      
      assertMatch("(!(a=c))", properties);
      assertMatch("(! (a=c))", properties);
      assertMatch("(!(a=c) )", properties);
      assertMatch("(!( a=c))", properties);
      assertMatch("(!(a =c))", properties);
      assertMatch("(!(a= c))", properties);
      assertMatch("(!(a=c ))", properties);
      assertNoMatch("(!(a=b))", properties);
      assertNoMatch("(! (a=b))", properties);
      assertNoMatch("(!( a=b))", properties);
      assertNoMatch("(!(a =b))", properties);
      assertNoMatch("(!(c= d))", properties);
      assertNoMatch("(!(e=f ))", properties);
      assertNoMatch("(!(a=b) )", properties);
      assertMatch("(!(x=c))", properties);
   }
   
   public void testAnd() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b")
      .put("c", "d")
      .put("e", "f");
      
      assertInvalid("(&)");
      assertInvalid("(&())");
      assertInvalid("(&()(a=b))");
      assertInvalid("(&(a=b)())");
      
      assertMatch("(&(a=b)(c=d)(e=f))", properties);
      assertMatch("(& (a=b)(c=d)(e=f))", properties);
      assertMatch("(&(a=b) (c=d)(e=f))", properties);
      assertMatch("(&(a=b)(c=d) (e=f))", properties);
      assertMatch("(&(a=b)(c=d)(e=f) )", properties);
      assertNoMatch("(&(a=x)(c=d)(e=f))", properties);
      assertNoMatch("(& (a=x)(c=d)(e=f))", properties);
      assertNoMatch("(&(a=x) (c=d)(e=f))", properties);
      assertNoMatch("(&(a=x)(c=d) (e=f))", properties);
      assertNoMatch("(&(a=x)(c=d)(e=f) )", properties);
      assertNoMatch("(&(a=b)(c=x)(e=f))", properties);
      assertNoMatch("(&(a=b)(c=d)(e=x))", properties);
      assertNoMatch("(&(a=b)(c=d)(e=f)(x=a))", properties);
   }
   
   public void testOr() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b")
      .put("c", "d")
      .put("e", "f");
      
      assertInvalid("(|)");
      assertInvalid("(|())");
      assertInvalid("(|()(a=b))");
      assertInvalid("(|(a=b)())");
      
      assertMatch("(|(a=b)(c=d)(e=f))", properties);
      assertMatch("(| (a=b)(c=d)(e=f))", properties);
      assertMatch("(|(a=b) (c=d)(e=f))", properties);
      assertMatch("(|(a=b)(c=d) (e=f))", properties);
      assertMatch("(|(a=b)(c=d)(e=f) )", properties);
      assertMatch("(|(a=x)(c=d)(e=f))", properties);
      assertMatch("(|(a=b)(c=x)(e=f))", properties);
      assertMatch("(|(a=b)(c=d)(e=x))", properties);
      assertMatch("(|(a=b)(c=d)(e=f)(x=a))", properties);
      assertMatch("(|(x=a)(a=b)(c=d)(e=f))", properties);
      assertNoMatch("(|(a=x)(c=x)(e=x))", properties);
      assertNoMatch("(| (a=x)(c=x)(e=x))", properties);
      assertNoMatch("(|(a=x) (c=x)(e=x))", properties);
      assertNoMatch("(|(a=x)(c=x) (e=x))", properties);
      assertNoMatch("(|(a=x)(c=x)(e=x) )", properties);
      assertNoMatch("(|(a=x)(c=x)(e=x)(x=a))", properties);
   }
   
   public void testCaseSensitive() throws Exception
   {
      PropertyBuilder properties = PropertyBuilder.build()
      .put("a", "b")
      .put("C", "d")
      .put("MiXeD", "Case");
      
      assertNoMatchCaseSensitive("(A=b)", properties);
      assertMatchNotCaseSensitive("(A=b)", properties);
      assertMatchCaseSensitive("(a=b)", properties);
      assertMatchNotCaseSensitive("(a=b)", properties);
      assertNoMatchCaseSensitive("(c=d)", properties);
      assertMatchNotCaseSensitive("(c=d)", properties);
      assertMatchCaseSensitive("(C=d)", properties);
      assertMatchNotCaseSensitive("(C=d)", properties);
      assertNoMatchCaseSensitive("(mixed=Case)", properties);
      assertMatchNotCaseSensitive("(mixed=Case)", properties);
      assertNoMatchCaseSensitive("(MIXED=Case)", properties);
      assertMatchNotCaseSensitive("(MIXED=Case)", properties);
      assertMatchCaseSensitive("(MiXeD=Case)", properties);
      assertMatchNotCaseSensitive("(MiXeD=Case)", properties);
   }

   protected void assertValid(String filter) throws Exception
   {
      createFilter(filter);
   }
   
   protected void assertInvalid(String filter) throws Exception
   {
      try
      {
         createFilter(filter);
         fail("Should not be here! filter=" + filter);
      }
      catch (Exception e)
      {
         // InvalidSyntaxException - If filter contains an invalid filter string that cannot be parsed. 
         // NullPointerException - If filter is null.
         checkThrowable(filter != null ? InvalidSyntaxException.class : NullPointerException.class, e);
      }
   }

   protected void assertMatch(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilter(filter, properties, true);
   }

   protected void assertNoMatch(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilter(filter, properties, false);
   }
      
   protected void assertFilter(String filter, PropertyBuilder properties, boolean expected) throws Exception
   {
      Filter f = createFilter(filter);
      assertFilter(f, properties, expected, false);
      assertFilter(f, properties, expected, true);
   }

   protected void assertFilter(Filter filter, PropertyBuilder properties, boolean expected, boolean caseSensitive) throws Exception
   {
      Dictionary<String, Object> dictionary = properties.getDictionary();
      boolean result;
      String description = "";
      if (caseSensitive)
      {
         result = filter.matchCase(dictionary);
         description = "(case sensitive)";
      }
      else
         result = filter.match(dictionary);

      getLog().debug(filter + " with " + dictionary + " result=" + result + " " + description);
      if (expected)
         assertTrue("Expected " + filter + " to match " + description + " with properties " + dictionary, result);
      else
         assertFalse("Expected " + filter + " to NOT match " + description + " with properties " + dictionary, result);
   }

   protected void assertMatchCaseSensitive(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilterCaseSensitive(filter, properties, true);
   }

   protected void assertNoMatchCaseSensitive(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilterCaseSensitive(filter, properties, false);
   }

   protected void assertMatchNotCaseSensitive(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilter(createFilter(filter), properties, true, false);
   }

   protected void assertNoMatchNotCaseSensitive(String filter, PropertyBuilder properties) throws Exception
   {
      assertFilter(createFilter(filter), properties, false, false);
   }

   protected void assertFilterCaseSensitive(String filter, PropertyBuilder properties, boolean expected) throws Exception
   {
      // Only the direct FilterImpl can do this
   }

   public static class PropertyBuilder
   {
      private Dictionary<String, Object> dictionary = new Hashtable<String, Object>();

      public static PropertyBuilder build()
      {
         return new PropertyBuilder();
      }
      
      public Dictionary<String, Object> getDictionary()
      {
         return dictionary;
      }
      
      public PropertyBuilder put(String key, Object value)
      {
         dictionary.put(key, value);
         return this;
      }
   }
}
