/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework;

//$Id: BundleContextImpl.java 92673 2009-08-21 15:09:34Z thomas.diesler@jboss.com $

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.EventListener;
import java.util.List;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.AbstractPlugin;
import org.jboss.osgi.jbossmc.api.BundleEventsPlugin;
import org.jboss.osgi.jbossmc.api.BundleLifecyclePlugin;
import org.jboss.osgi.jbossmc.api.BundleRegistryPlugin;
import org.jboss.osgi.jbossmc.api.BundleStoragePlugin;
import org.jboss.osgi.jbossmc.api.FrameworkEventsPlugin;
import org.jboss.osgi.jbossmc.api.ServiceEventsPlugin;
import org.jboss.osgi.jbossmc.api.ServiceRegistryPlugin;
import org.jboss.osgi.spi.NotImplementedException;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.BundleListener;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkListener;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.launch.Framework;

/**
 * An implementation of an OSGi BundleContext
 * 
 * @author thomas.diesler@jboss.com
 * @since 29-Jul-2009
 */
public class BundleContextImpl implements BundleContext
{
   // Provide logging
   final Logger log = Logger.getLogger(BundleContextImpl.class);

   private Framework framework;
   private Bundle bundle;
   
   private List<EventListener> registeredListeners = new ArrayList<EventListener>();
   private List<ServiceRegistration> registeredServices = new ArrayList<ServiceRegistration>();
   private List<ServiceReference> usedServices = new ArrayList<ServiceReference>();
   
   public BundleContextImpl(Framework framework, Bundle bundle)
   {
      this.framework = framework;
      this.bundle = bundle;
   }

   public Framework getFramework()
   {
      return framework;
   }
   
   public <T extends AbstractPlugin> T getPlugin(Class<T> clazz)
   {
      FrameworkImpl frameworkImpl = (FrameworkImpl)framework;
      return frameworkImpl.getPlugin(clazz);
   }
   
   public void addBundleListener(BundleListener listener)
   {
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.addBundleListener(listener);
      registeredListeners.add(listener);
   }

   public void removeBundleListener(BundleListener listener)
   {
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.removeBundleListener(listener);
      registeredListeners.remove(listener);
   }

   public void addFrameworkListener(FrameworkListener listener)
   {
      FrameworkEventsPlugin eventManager = getPlugin(FrameworkEventsPlugin.class);
      eventManager.addFrameworkListener(listener);
      registeredListeners.add(listener);
   }

   public void removeFrameworkListener(FrameworkListener listener)
   {
      FrameworkEventsPlugin eventManager = getPlugin(FrameworkEventsPlugin.class);
      eventManager.removeFrameworkListener(listener);
      registeredListeners.remove(listener);
   }

   public void addServiceListener(ServiceListener listener)
   {
      ServiceEventsPlugin eventManager = getPlugin(ServiceEventsPlugin.class);
      eventManager.addServiceListener(listener);
      registeredListeners.add(listener);
   }

   public void addServiceListener(ServiceListener listener, String filter) throws InvalidSyntaxException
   {
      ServiceEventsPlugin eventManager = getPlugin(ServiceEventsPlugin.class);
      eventManager.addServiceListener(listener, filter);
      registeredListeners.add(listener);
   }

   public void removeServiceListener(ServiceListener listener)
   {
      ServiceEventsPlugin eventManager = getPlugin(ServiceEventsPlugin.class);
      eventManager.removeServiceListener(listener);
      registeredListeners.remove(listener);
   }

   public Filter createFilter(String filter) throws InvalidSyntaxException
   {
      return FrameworkUtil.createFilter(filter);
   }

   public Bundle getBundle()
   {
      return bundle;
   }

   public Bundle getBundle(long id)
   {
      BundleRegistryPlugin bundleRegistry = getPlugin(BundleRegistryPlugin.class);
      return bundleRegistry.getBundleById(id);
   }

   public Bundle[] getBundles()
   {
      BundleRegistryPlugin bundleRegistry = getPlugin(BundleRegistryPlugin.class);
      List<Bundle> bundleList = bundleRegistry.getBundles();
      Bundle[] bundles = new Bundle[bundleList.size()];
      return bundleList.toArray(bundles);
   }

   public File getDataFile(String filename)
   {
      BundleStoragePlugin bundleStorage = getPlugin(BundleStoragePlugin.class);
      return bundleStorage.getDataFile(bundle, filename);
   }

   public String getProperty(String key)
   {
      FrameworkImpl frameworkImpl = (FrameworkImpl)framework;
      String property = (String)frameworkImpl.getProperty(key);
      if (property == null)
         property = System.getProperty(key);
      
      return property;
   }

   public Object getService(ServiceReference reference)
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      Object service = serviceRegistry.getService(bundle, reference);
      if (service != null)
      {
         usedServices.add(reference);
      }
      return service;
   }

   public boolean ungetService(ServiceReference reference)
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      return serviceRegistry.ungetService(bundle, reference);
   }
   
   public ServiceReference getServiceReference(String clazz)
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      return serviceRegistry.getServiceReference(clazz);
   }

   public ServiceReference[] getServiceReferences(String clazz, String filter) throws InvalidSyntaxException
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      return serviceRegistry.getServiceReferences(clazz, filter);
   }

   public ServiceReference[] getAllServiceReferences(String clazz, String filter) throws InvalidSyntaxException
   {
      // [TODO] BundleContext.getAllServiceReferences(String clazz, String filter)
      throw new NotImplementedException();
   }

   public Bundle installBundle(String location) throws BundleException
   {
      BundleLifecyclePlugin plugin = getPlugin(BundleLifecyclePlugin.class);
      return plugin.installBundle(location);
   }

   public Bundle installBundle(String location, InputStream input) throws BundleException
   {
      // [TODO] BundleContext.installBundle(String location, InputStream input)
      throw new NotImplementedException();
   }

   @SuppressWarnings("unchecked")
   public ServiceRegistration registerService(String[] clazzes, Object service, Dictionary properties)
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      ServiceRegistration sreg = serviceRegistry.registerService(bundle, clazzes, service, properties);
      registeredServices.add(sreg);
      return sreg;
   }

   @SuppressWarnings("unchecked")
   public ServiceRegistration registerService(String clazz, Object service, Dictionary properties)
   {
      ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
      ServiceRegistration sreg = serviceRegistry.registerService(bundle, clazz, service, properties);
      registeredServices.add(sreg);
      return sreg;
   }

   public void releaseRegisteredListeners()
   {
      ArrayList<EventListener> reverseList = new ArrayList<EventListener>(registeredListeners);
      Collections.reverse(reverseList);
      
      for (EventListener listener : reverseList)
      {
         try
         {
            if (listener instanceof FrameworkListener)
               removeFrameworkListener((FrameworkListener)listener);
            else if (listener instanceof ServiceListener)
               removeServiceListener((ServiceListener)listener);
            else if (listener instanceof BundleListener)
               removeBundleListener((BundleListener)listener);
         }
         catch (Exception ex)
         {
            log.error("Cannot release registered listener", ex);
         }
      }
      registeredListeners.clear();
   }

   public void releaseRegisteredServices()
   {
      ArrayList<ServiceRegistration> reverseList = new ArrayList<ServiceRegistration>(registeredServices);
      Collections.reverse(reverseList);
      
      for (ServiceRegistration sreg : reverseList)
      {
         try
         {
            sreg.unregister();
         }
         catch (Exception ex)
         {
            log.error("Cannot release registered service", ex);
         }
      }
      registeredServices.clear();
   }

   public void releaseUsedServices()
   {
      ArrayList<ServiceReference> reverseList = new ArrayList<ServiceReference>(usedServices);
      Collections.reverse(reverseList);
      
      for (ServiceReference sref : reverseList)
      {
         try
         {
            ungetService(sref);
         }
         catch (Exception ex)
         {
            log.error("Cannot release used service", ex);
         }
      }
      usedServices.clear();
   }

   @Override
   public String toString()
   {
      return "BundleContext " + bundle;
   }
}