/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework;

//$Id: BundleImpl.java 92613 2009-08-20 16:35:19Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.cert.X509Certificate;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.AbstractPlugin;
import org.jboss.osgi.jbossmc.api.BundleLifecyclePlugin;
import org.jboss.osgi.jbossmc.api.BundleResolverPlugin;
import org.jboss.osgi.jbossmc.api.FrameworkEventsPlugin;
import org.jboss.osgi.jbossmc.api.RuntimeClassLoader;
import org.jboss.osgi.spi.NotImplementedException;
import org.jboss.virtual.VirtualFile;
import org.jboss.virtual.VirtualFileFilter;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkEvent;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.Version;

/**
 * An implementation of an OSGi Bundle
 * 
 * @author thomas.diesler@jboss.com
 * @since 29-Jul-2009
 */
public class BundleImpl implements Bundle
{
   // Provide logging
   final Logger log = Logger.getLogger(BundleImpl.class);

   private int state;
   private VirtualFile vFile;
   private long bundleId;
   private BundleContext context;
   private Dictionary<String, String> headers;
   private RuntimeClassLoader runtimeLoader;
   private BundleActivator activator;
   private String toStringCache;
   
   public BundleImpl(VirtualFile vFile)
   {
      this.vFile = vFile;
   }

   public VirtualFile getVirtualFile()
   {
      return vFile;
   }

   public long getBundleId()
   {
      return bundleId;
   }

   public void setBundleId(long bundleId)
   {
      this.bundleId = bundleId;
   }

   public int getState()
   {
      return state;
   }

   public void setState(int state)
   {
      this.state = state;
   }

   public String getLocation()
   {
      try
      {
         return getVirtualFile().toURL().toExternalForm();
      }
      catch (Exception ex)
      {
         throw new IllegalStateException("Cannot obtain bundle location", ex);
      }
   }

   public String getSymbolicName()
   {
      return getHeader(Constants.BUNDLE_SYMBOLICNAME);
   }

   public BundleActivator getBundleActivator()
   {
      return activator;
   }

   public void setBundleActivator(BundleActivator activator)
   {
      this.activator = activator;
   }

   private RuntimeClassLoader getRuntimeClassLoader()
   {
      return runtimeLoader;
   }

   public void setRuntimeClassLoader(RuntimeClassLoader runtimeLoader)
   {
      this.runtimeLoader = runtimeLoader;
   }

   public void start() throws BundleException
   {
      BundleLifecyclePlugin bundleLifecycle = getPlugin(BundleLifecyclePlugin.class);
      bundleLifecycle.start(this, 0);
   }

   public void start(int options) throws BundleException
   {
      BundleLifecyclePlugin bundleLifecycle = getPlugin(BundleLifecyclePlugin.class);
      bundleLifecycle.start(this, options);
   }

   public void stop() throws BundleException
   {
      BundleLifecyclePlugin bundleLifecycle = getPlugin(BundleLifecyclePlugin.class);
      bundleLifecycle.stop(this, 0);
   }

   public void stop(int options) throws BundleException
   {
      BundleLifecyclePlugin bundleLifecycle = getPlugin(BundleLifecyclePlugin.class);
      bundleLifecycle.stop(this, options);
   }

   public void uninstall() throws BundleException
   {
      BundleLifecyclePlugin bundleLifecycle = getPlugin(BundleLifecyclePlugin.class);
      bundleLifecycle.uninstall(this);
   }

   public void update() throws BundleException
   {
      // [TODO] Bundle.update()
      throw new NotImplementedException();
   }

   public void update(InputStream in) throws BundleException
   {
      // [TODO] Bundle.update(InputStream in)
      throw new NotImplementedException();
   }

   public BundleContext getBundleContext()
   {
      return context;
   }

   public void setBundleContext(BundleContextImpl bndContext)
   {
      this.context = bndContext;
   }

   public URL getEntry(String path)
   {
      assertNotUninstalled();

      URL entryURL = null;
      try
      {
         VirtualFile child = vFile.getChild(path);
         if (child != null)
            entryURL = child.toURL();
      }
      catch (Exception ex)
      {
         // ignore
      }
      return entryURL;
   }

   public Enumeration<String> getEntryPaths(String path)
   {
      assertNotUninstalled();

      Vector<String> paths = new Vector<String>();
      try
      {
         VirtualFile child = vFile.getChild(path);
         if (child != null)
         {
            for (VirtualFile vf : child.getChildrenRecursively())
            {
               String entry = vf.getPathName();
               if (vf.isLeaf() == false)
                  entry = entry + "/";
               
               paths.add(entry);
            }
         }
      }
      catch (Exception ex)
      {
         // ignore
      }
      return paths.size() > 0 ? paths.elements() : null;
   }

   public Enumeration<URL> findEntries(String path, String filePattern, boolean recurse)
   {
      assertNotUninstalled();

      // If null is specified, this is equivalent to "*" and matches all files.
      if (filePattern == null)
         filePattern = "*";

      final Filter filter;
      try
      {
         filter = FrameworkUtil.createFilter("(name=" + filePattern + ")");
      }
      catch (InvalidSyntaxException e)
      {
         throw new IllegalArgumentException("Cannot create filter from: " + filePattern);
      }
      
      final Hashtable<String, String> dictionary = new Hashtable<String, String>();
      VirtualFileFilter vfFilter = new VirtualFileFilter()
      {
         public boolean accepts(VirtualFile file)
         {
            String name = file.getName();
            dictionary.put("name", name);
            boolean match = filter.match(dictionary);
            return match;
         }
      };

      Vector<URL> paths = new Vector<URL>();
      try
      {
         VirtualFile child = vFile.getChild(path);
         if (child != null)
         {
            List<VirtualFile> matchedFiles = (recurse ? child.getChildrenRecursively(vfFilter) : child.getChildren(vfFilter));
            for (VirtualFile vf : matchedFiles)
            {
               paths.add(vf.toURL());
            }
         }
      }
      catch (Exception ex)
      {
         // ignore
      }
      return paths.size() > 0 ? paths.elements() : null;
   }

   public Dictionary<String, String> getHeaders()
   {
      if (headers == null)
      {
         headers = new Hashtable<String, String>();
         try
         {
            VirtualFile child = vFile.getChild("META-INF/MANIFEST.MF");
            Manifest manifest = new Manifest(child.openStream());
            Attributes attribs = manifest.getMainAttributes();
            for (Object key : attribs.keySet())
            {
               Object value = attribs.get(key);
               headers.put(key.toString(), value.toString());
            }
         }
         catch (IOException ex)
         {
            throw new IllegalStateException("Cannot read bundle manifest", ex);
         }
      }
      return headers;
   }

   public Dictionary<String, String> getHeaders(String locale)
   {
      // [TODO] Bundle.getHeaders(String locale)
      throw new NotImplementedException();
   }

   public long getLastModified()
   {
      // [TODO] Bundle.getLastModified()
      throw new NotImplementedException();
   }

   public ServiceReference[] getRegisteredServices()
   {
      // [TODO] Bundle.getRegisteredServices()
      throw new NotImplementedException();
   }

   public ServiceReference[] getServicesInUse()
   {
      // [TODO] Bundle.getServicesInUse()
      throw new NotImplementedException();
   }

   public URL getResource(String name)
   {
      return getRuntimeClassLoader().getResource(name);
   }

   public Enumeration<URL> getResources(String name) throws IOException
   {
      return getRuntimeClassLoader().getResources(name);
   }

   public Map<X509Certificate, List<X509Certificate>> getSignerCertificates(int signersType)
   {
      // [TODO] Bundle.getSignerCertificates(int signersType)
      throw new NotImplementedException();
   }

   public Version getVersion()
   {
      String versionStr = getHeader(Constants.BUNDLE_VERSION);
      return new Version(versionStr);
   }

   public boolean hasPermission(Object permission)
   {
      // [TODO] Bundle.hasPermission(Object permission)
      throw new NotImplementedException();
   }

   public Class<?> loadClass(String name) throws ClassNotFoundException
   {
      // [TODO] If this bundle is a fragment bundle then this method must throw a ClassNotFoundException

      // If this bundle's state is INSTALLED, this method must attempt to resolve this bundle 
      // before attempting to load the class.
      if (state == Bundle.INSTALLED)
      {
         BundleResolverPlugin resolver = getPlugin(BundleResolverPlugin.class);
         try
         {
            resolver.resolveBundle(this);
         }
         catch (BundleException ex)
         {
            // If this bundle cannot be resolved, a Framework event of type FrameworkEvent.ERROR is fired
            FrameworkEventsPlugin eventManager = getPlugin(FrameworkEventsPlugin.class);
            eventManager.fireFrameworkEvent(new FrameworkEvent(FrameworkEvent.ERROR, this, ex));

            throw new ClassNotFoundException("Cannot load class: " + name, ex);
         }
      }

      return getRuntimeClassLoader().loadClass(name);
   }

   private String getHeader(String header)
   {
      return (String)getHeaders().get(header);
   }

   private <T extends AbstractPlugin> T getPlugin(Class<T> clazz)
   {
      BundleContextImpl contextImpl = (BundleContextImpl)context;
      return contextImpl.getPlugin(clazz);
   }
   
   private void assertNotUninstalled()
   {
      if (state == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already uninstalled: " + this);
   }

   @Override
   public boolean equals(Object obj)
   {
      if (!(obj instanceof BundleImpl))
         return false;

      BundleImpl other = (BundleImpl)obj;
      return other.getBundleId() == getBundleId();
   }

   @Override
   public int hashCode()
   {
      return new Long(getBundleId()).hashCode();
   }

   @Override
   public String toString()
   {
      if (toStringCache == null || getBundleId() == 0)
         toStringCache = "[" + getBundleId() + "] " + getSymbolicName() + "-" + getVersion();

      return toStringCache;
   }
}