/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework;

//$Id: FrameworkImpl.java 92648 2009-08-21 01:30:02Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.CodeSource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.kernel.Kernel;
import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.AbstractPlugin;
import org.jboss.osgi.jbossmc.api.AutoInstallPlugin;
import org.jboss.osgi.jbossmc.api.BundleEventsPlugin;
import org.jboss.osgi.jbossmc.api.BundleRegistryPlugin;
import org.jboss.osgi.jbossmc.api.BundleStoragePlugin;
import org.jboss.osgi.jbossmc.api.FrameworkEventsPlugin;
import org.jboss.osgi.jbossmc.api.ServiceEventsPlugin;
import org.jboss.osgi.jbossmc.api.ServiceRegistryPlugin;
import org.jboss.osgi.jbossmc.framework.classloading.BundleResolverPluginImpl;
import org.jboss.osgi.jbossmc.framework.classloading.ClassLoaderFactoryPluginImpl;
import org.jboss.osgi.jbossmc.framework.classloading.FrameworkClassLoader;
import org.jboss.osgi.jbossmc.framework.plugins.BundleEventsPluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.BundleFactoryPluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.BundleLifecyclePluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.BundleRegistryPluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.BundleStoragePluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.FrameworkEventsPluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.ServiceEventsPluginImpl;
import org.jboss.osgi.jbossmc.framework.plugins.ServiceRegistryPluginImpl;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkEvent;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.launch.Framework;

/**
 * An impementation of an OSGi Framework
 * 
 * @author thomas.diesler@jboss.com
 * @since 29-Jul-2009
 */
public class FrameworkImpl extends BundleImpl implements Framework
{
   // Provide logging
   final Logger log = Logger.getLogger(FrameworkImpl.class);

   private Map<String, Object> properties = new HashMap<String, Object>();
   private Map<Class<?>, AbstractPlugin> plugins = new HashMap<Class<?>, AbstractPlugin>();
   private ClassLoader frameworkLoader;
   private int startLevel;
   private Kernel kernel;

   public FrameworkImpl(Kernel kernel, Map<String, Object> props)
   {
      super(getFrameworkCodeSource());

      if (kernel == null)
         throw new IllegalArgumentException("Kernel cannot be null");

      this.kernel = kernel;

      if (props != null)
         properties.putAll(props);
      
      frameworkLoader = new FrameworkClassLoader(this);

      // Initialize default plugins
      addPlugin(new BundleEventsPluginImpl(this));
      addPlugin(new BundleFactoryPluginImpl(this));
      addPlugin(new BundleLifecyclePluginImpl(this));
      addPlugin(new BundleResolverPluginImpl(this));
      addPlugin(new BundleRegistryPluginImpl(this));
      addPlugin(new BundleStoragePluginImpl(this));
      addPlugin(new ClassLoaderFactoryPluginImpl(this));
      addPlugin(new FrameworkEventsPluginImpl(this));
      addPlugin(new ServiceEventsPluginImpl(this));
      addPlugin(new ServiceRegistryPluginImpl(this));
   }
   
   public Kernel getKernel()
   {
      return kernel;
   }

   @SuppressWarnings("unchecked")
   public <T extends AbstractPlugin> T getPlugin(Class<T> clazz)
   {
      T plugin = (T)plugins.get(clazz);
      if (plugin == null)
         throw new IllegalStateException("Cannot obtain plugin for: " + clazz.getName());

      return plugin;
   }

   @SuppressWarnings("unchecked")
   public <T extends AbstractPlugin> T getOptionalPlugin(Class<T> clazz)
   {
      return (T)plugins.get(clazz);
   }

   public void addPlugin(AbstractPlugin plugin)
   {
      Class<? extends AbstractPlugin> clazz = plugin.getClass();
      for (Class<?> interf : clazz.getInterfaces())
      {
         if (AbstractPlugin.class.isAssignableFrom(interf))
         {
            log.debug("Add plugin: " + clazz.getName());
            plugins.put(interf, plugin);
         }
      }
   }

   public ClassLoader getFrameworkLoader()
   {
      return frameworkLoader;
   }

   /**
    * Returns the Framework unique identifier. This Framework is assigned the unique identifier zero (0) since this Framework is also a System Bundle.
    */
   @Override
   public long getBundleId()
   {
      return 0;
   }

   @Override
   public String getLocation()
   {
      return "System Bundle";
   }

   /**
    * Returns the symbolic name of this Framework.
    */
   @Override
   public String getSymbolicName()
   {
      return "org.jboss.osgi.jbossmc";
   }

   int getStartLevel()
   {
      return startLevel;
   }

   void setStartLevel(int startLevel)
   {
      this.startLevel = startLevel;
   }

   public Object getProperty(String key)
   {
      return properties.get(key);
   }

   public void init() throws BundleException
   {
      initInternal();
   }

   @Override
   public void start() throws BundleException
   {
      startInternal();
   }

   @Override
   public void start(int options) throws BundleException
   {
      startInternal();
   }

   @Override
   public void stop() throws BundleException
   {
      // [TODO] The method returns immediately to the caller after initiating the following steps

      stopInternal();
   }

   @Override
   public void stop(int options) throws BundleException
   {
      // [TODO] The method returns immediately to the caller after initiating the following steps

      stopInternal();
   }

   @Override
   public void update() throws BundleException
   {
      updateInternal();
   }

   /**
    * Calling this method is the same as calling {@link #update()} except that any provided InputStream is immediately closed.
    */
   @Override
   public void update(InputStream in) throws BundleException
   {
      if (in != null)
      {
         try
         {
            in.close();
         }
         catch (IOException ex)
         {
            // ignore
         }
      }

      // [TODO] The method returns immediately to the caller after initiating the following steps

      updateInternal();
   }

   /**
    * The Framework cannot be uninstalled.
    * <p>
    * This method always throws a BundleException.
    */
   @Override
   public void uninstall() throws BundleException
   {
      throw new BundleException("The system bundle cannot be uninstalled");
   }

   private void initInternal()
   {
      // This method does nothing if called when this Framework is in the
      // STARTING, ACTIVE or STOPPING state
      if (getState() == STARTING || getState() == ACTIVE || getState() == STOPPING)
         return;

      // Put into the STARTING state
      setState(STARTING);

      // Have a valid Bundle Context
      setBundleContext(new SystemBundleContext(this));

      // Be at start level 0
      setStartLevel(0);

      // Have event handling enabled
      FrameworkEventsPlugin frameworkEvents = getPlugin(FrameworkEventsPlugin.class);
      frameworkEvents.setEnabled(true);
      BundleEventsPlugin bundleEvents = getPlugin(BundleEventsPlugin.class);
      bundleEvents.setEnabled(true);
      ServiceEventsPlugin serviceEvents = getPlugin(ServiceEventsPlugin.class);
      serviceEvents.setEnabled(true);

      // Cleanup the storage area
      String storageClean = (String)getProperty(Constants.FRAMEWORK_STORAGE_CLEAN);
      if (storageClean != null)
      {
         BundleStoragePlugin storage = getPlugin(BundleStoragePlugin.class);
         storage.cleanStorage(storageClean);
      }
      
      // [TODO] Have reified Bundle objects for all installed bundles

      // [TODO] Have registered any framework services
   }

   private void startInternal() throws BundleException
   {
      // If this Framework is not in the STARTING state, initialize this Framework
      if (getState() != STARTING)
         initInternal();

      // All installed bundles must be started
      AutoInstallPlugin autoInstall = getOptionalPlugin(AutoInstallPlugin.class);
      if (autoInstall != null)
      {
         autoInstall.installBundles();
         autoInstall.startBundles();
      }

      // This Framework's state is set to ACTIVE
      setState(ACTIVE);

      // A framework event of type STARTED is fired
      FrameworkEventsPlugin frameworkEvents = getPlugin(FrameworkEventsPlugin.class);
      frameworkEvents.fireFrameworkEvent(new FrameworkEvent(FrameworkEvent.STARTED, this, null));
   }

   private void stopInternal()
   {
      // This Framework's state is set to STOPPING
      setState(STOPPING);

      // All installed bundles must be stopped
      BundleRegistryPlugin bundleRegistry = getPlugin(BundleRegistryPlugin.class);
      List<Bundle> reverseList = new ArrayList<Bundle>(bundleRegistry.getBundles());
      Collections.reverse(reverseList);
      for (Bundle bundle : reverseList)
      {
         try
         {
            bundle.stop();
         }
         catch (BundleException ex)
         {
            log.error("Cannot stop bundle: " + bundle, ex);
         }
      }

      // Unregister all services registered by this Framework
      ServiceReference[] srefs = null;
      try
      {
         ServiceRegistryPlugin serviceRegistry = getPlugin(ServiceRegistryPlugin.class);
         srefs = serviceRegistry.getServiceReferences(null, null);
      }
      catch (InvalidSyntaxException ex)
      {
         // ignore
      }
      if (srefs != null)
      {
         for (ServiceReference sref : srefs)
         {
            ServiceReferenceImpl srefImpl = (ServiceReferenceImpl)sref;
            try
            {
               srefImpl.getServiceRegistration().unregister();
            }
            catch (Exception ex)
            {
               log.error("Cannot unregister service: " + sref, ex);
            }
         }
      }

      // Event handling is disabled
      ServiceEventsPlugin serviceEvents = getPlugin(ServiceEventsPlugin.class);
      serviceEvents.setEnabled(false);
      BundleEventsPlugin bundleEvents = getPlugin(BundleEventsPlugin.class);
      bundleEvents.setEnabled(false);
      FrameworkEventsPlugin frameworkEvents = getPlugin(FrameworkEventsPlugin.class);
      frameworkEvents.setEnabled(false);

      // This Framework's state is set to RESOLVED
      setState(RESOLVED);

      // [TODO] All resources held by this Framework are released.

      // [TODO] Notify all threads that are waiting at waitForStop(long)
   }

   private void updateInternal() throws BundleException
   {
      stopInternal();

      startInternal();
   }

   public FrameworkEvent waitForStop(long timeout) throws InterruptedException
   {
      // [TODO] Wait until this Framework has completely stopped.

      // [TODO] A Framework Event indicating the reason this method returned
      return new FrameworkEvent(FrameworkEvent.STOPPED, this, null);
   }

   private static VirtualFile getFrameworkCodeSource()
   {
      CodeSource codeSource = FrameworkImpl.class.getProtectionDomain().getCodeSource();
      URL locationURL = codeSource.getLocation();
      VirtualFile vfsRoot;
      try
      {
         vfsRoot = VFS.createNewRoot(locationURL);
      }
      catch (IOException ex)
      {
         throw new IllegalArgumentException("Invalid framework location URL: " + locationURL);
      }
      return vfsRoot;
   }
   
   @Override
   public String toString()
   {
      return "[" + getBundleId() + "] " + getSymbolicName();
   }
}