/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework;

//$Id: $

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.logging.Logger;
import org.jboss.osgi.spi.NotImplementedException;
import org.osgi.framework.Bundle;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceFactory;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;

/**
 * A simple implementation of a ServiceReference
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Aug-2009
 */
public class ServiceReferenceImpl implements ServiceReference
{
   // Provide logging
   final Logger log = Logger.getLogger(ServiceReferenceImpl.class);

   private Bundle bundle;
   private Object service;
   private Dictionary<String, Object> properties;
   private ServiceRegistration registration;
   private Map<Bundle, Long> bundleUseCount = new HashMap<Bundle, Long>();
   private Map<Bundle, Object> cachedServices = new HashMap<Bundle, Object>();
   private boolean unregistered;
   private String toStringCache;
   
   public ServiceReferenceImpl(Bundle bundle, Object service, Dictionary<String, Object> props)
   {
      this.bundle = bundle;
      this.service = service;
      this.properties = props;
   }

   public Long getUseCount(Bundle bundle)
   {
      Long useCount = bundleUseCount.get(bundle);
      return useCount != null ? useCount : new Long(0);
   }
   
   public Long incrementUseCount(Bundle bundle)
   {
      Long useCount = new Long(getUseCount(bundle) + 1);
      bundleUseCount.put(bundle, useCount);
      return useCount;
   }
   
   public Long decrementUseCount(Bundle bundle)
   {
      Long useCount = new Long(getUseCount(bundle) - 1);
      bundleUseCount.put(bundle, useCount);
      return useCount;
   }
   
   public Object getCachedService(Bundle bundle)
   {
      return cachedServices.get(bundle);
   }

   public void setCachedService(Bundle bundle, Object service)
   {
      if (service != null)
         cachedServices.put(bundle, service);
      else
         cachedServices.remove(bundle);
   }

   public boolean isUnregistered()
   {
      return unregistered;
   }

   public void setUnregistered(boolean unregistered)
   {
      this.unregistered = unregistered;
   }

   public Object getService()
   {
      return service;
   }
   
   public ServiceRegistration getServiceRegistration()
   {
      return registration;
   }

   public void setServiceRegistration(ServiceRegistration serviceRegistration)
   {
      this.registration = serviceRegistration;
   }

   public int compareTo(Object reference)
   {
      // [TODO] ServiceReference.compareTo(Object reference)
      throw new NotImplementedException();
   }

   public Bundle getBundle()
   {
      return bundle;
   }

   public Object getProperty(String key)
   {
      return properties.get(key);
   }

   public String[] getPropertyKeys()
   {
      Enumeration<String> keysEnum = properties.keys();
      List<String> keyList = new ArrayList<String>();
      while (keysEnum.hasMoreElements())
         keyList.add(keysEnum.nextElement());
      
      String[] keys = new String[keyList.size()];
      keyList.toArray(keys);
      
      return keys;
   }

   public Bundle[] getUsingBundles()
   {
      // [TODO] ServiceReference.getUsingBundles()
      throw new NotImplementedException();
   }

   public boolean isAssignableTo(Bundle bundle, String className)
   {
      // [TODO] ServiceReference.isAssignableTo(Bundle bundle, String className)
      throw new NotImplementedException();
   }

   public void unregister()
   {
      // For each bundle whose use count for this service is greater than zero:
      // 1. The bundle's use count for this service is set to zero.
      // 2. If the service was registered with a ServiceFactory object, the ServiceFactory.ungetService method is called to release the service object for the bundle.
      for (Bundle bundle : bundleUseCount.keySet())
      {
         Long count = bundleUseCount.get(bundle);
         if (count > 0)
         {
            if (service instanceof ServiceFactory)
            {
               ServiceFactory factory = (ServiceFactory)service;
               Object cachedService = cachedServices.remove(bundle);
               factory.ungetService(bundle, registration, cachedService);
            }
         }
      }
      
      bundleUseCount = null;
      cachedServices = null;
   }
   
   @Override
   public String toString()
   {
      if (toStringCache == null)
      {
         Long id = (Long)properties.get(Constants.SERVICE_ID);
         StringBuffer buffer = new StringBuffer("[" + id + "] ");
         buffer.append(Arrays.asList((String[])properties.get(Constants.OBJECTCLASS)));
         
         for (String key : getPropertyKeys())
         {
            if (!key.equals(Constants.SERVICE_ID) && !key.equals(Constants.OBJECTCLASS))
            {
               buffer.append("\n " + key + "=" + properties.get(key));
            }
         }
         toStringCache = buffer.toString();
      }
      return toStringCache;
   }
}