/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework.classloading;

// $Id: $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.framework.FrameworkImpl;
import org.osgi.framework.Constants;

/**
 * A ClassLoader that only loads system packages defined by the framework.
 * 
 * @author thomas.diesler@jboss.com
 * @since 29-Jul-2009
 */
public class FrameworkClassLoader extends ClassLoader
{
   // Provide logging
   final Logger log = Logger.getLogger(FrameworkClassLoader.class);

   private List<String> sysPackagesFull = new ArrayList<String>();
   private List<String> sysPackages = new ArrayList<String>();

   public FrameworkClassLoader(FrameworkImpl framework)
   {
      // Initialize system packages
      String sysPackageStr = (String)framework.getProperty(Constants.FRAMEWORK_SYSTEMPACKAGES);
      if (sysPackageStr == null)
      {
         sysPackagesFull.add("org.osgi.framework");
         sysPackagesFull.add("org.osgi.service.startlevel");
         sysPackagesFull.add("org.osgi.service.packageadmin");
         sysPackagesFull.add("org.osgi.util.tracker");
      }
      else
      {
         sysPackagesFull.clear();
         sysPackagesFull.addAll(toList(sysPackageStr));
      }

      // Add extra system packages
      String sysPackagesExtra = (String)framework.getProperty(Constants.FRAMEWORK_SYSTEMPACKAGES_EXTRA);
      if (sysPackagesExtra != null)
      {
         sysPackagesFull.addAll(toList(sysPackagesExtra));
      }

      // Remove the version specifier 
      for (String packName : sysPackagesFull)
      {
         int semicolon = packName.indexOf(';');
         if (semicolon > 0)
            packName = packName.substring(0, semicolon);

         sysPackages.add(packName.trim());
      }

      log.debug("Framework System Packages");
      ArrayList<String> sortedList = new ArrayList<String>(sysPackagesFull);
      Collections.sort(sortedList);
      for (String packName : sortedList)
         log.debug("  [" + packName + "]");
   }

   private List<String> toList(String packagesStr)
   {
      List<String> list = new ArrayList<String>();
      StringTokenizer st = new StringTokenizer(packagesStr, ",");
      while (st.hasMoreTokens())
      {
         String token = st.nextToken();
         list.add(token.trim());
      }
      return list;
   }

   @Override
   public Class<?> loadClass(String name) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Cannot load null class");
      
      String packPrefix = name.substring(0, name.lastIndexOf('.'));
      if (!name.startsWith("java.") && !sysPackages.contains(packPrefix))
         throw new ClassNotFoundException("Cannot load from system packages: " + name);

      return super.loadClass(name);
   }
}