/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework.classloading;

//$Id: $

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.BundleStoragePlugin;
import org.jboss.osgi.jbossmc.api.RuntimeClassLoader;
import org.jboss.osgi.jbossmc.framework.BundleContextImpl;
import org.jboss.osgi.jbossmc.framework.BundleImpl;
import org.jboss.osgi.jbossmc.framework.FrameworkImpl;
import org.jboss.osgi.spi.NotImplementedException;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;
import org.osgi.framework.Bundle;
import org.osgi.framework.launch.Framework;

/**
 * An implementation of a RuntimeClassLoader that uses a flat class space for all installed bundles.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Aug-2009
 */
public class RuntimeClassLoaderImpl implements RuntimeClassLoader
{
   // Provide logging
   final static Logger log = Logger.getLogger(RuntimeClassLoaderImpl.class);

   private static GlobalFrameworkLoader globalClassLoader;

   private Bundle bundle;
   private VirtualFile vFile;
   private ClassLoader parentClassLoader;

   public RuntimeClassLoaderImpl(Framework framework, Bundle bundle)
   {
      this.bundle = bundle;
      this.vFile = ((BundleImpl)bundle).getVirtualFile();

      parentClassLoader = getClass().getClassLoader();

      if (globalClassLoader == null)
      {
         FrameworkImpl frameworkImpl = (FrameworkImpl)framework;
         ClassLoader frameworkLoader = frameworkImpl.getFrameworkLoader();
         globalClassLoader = new GlobalFrameworkLoader(frameworkLoader);
      }

      try
      {
         addClassLoaderURL(vFile);
      }
      catch (IOException ex)
      {
         throw new IllegalStateException("Cannot add bundle url", ex);
      }
   }

   private void addClassLoaderURL(VirtualFile vFile) throws IOException
   {
      BundleContextImpl context = (BundleContextImpl)bundle.getBundleContext();
      BundleStoragePlugin storage = context.getPlugin(BundleStoragePlugin.class);

      File storageFile = storage.getDataFile(bundle, "internal/" + vFile.getName());
      VFSUtils.copyStreamAndClose(vFile.openStream(), new FileOutputStream(storageFile));

      globalClassLoader.addURL(storageFile.toURL());
      for (VirtualFile child : vFile.getChildren())
      {
         if (child.isArchive())
         {
            addClassLoaderURL(child);
         }
      }
   }

   public Class<?> loadClass(String name) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Cannot load null class");

      // Step #1 - class or resource is in a java.* package
      if (name.startsWith("java."))
         return parentClassLoader.loadClass(name);

      return globalClassLoader.loadClass(name);
   }

   public URL getResource(String name)
   {
      URL resourceURL = null;
      try
      {
         VirtualFile child = vFile.getChild(name);
         if (child != null)
            resourceURL = child.toURL();
      }
      catch (Exception ex)
      {
         // ignore
      }
      return resourceURL;
   }

   public Enumeration<URL> getResources(String name) throws IOException
   {
      // [TODO] Bundle.getResources(String name)
      throw new NotImplementedException();
   }

   static class GlobalFrameworkLoader extends URLClassLoader
   {
      private ClassLoader parent;
      private Set<String> urlCache = new HashSet<String>();

      public GlobalFrameworkLoader(ClassLoader parent)
      {
         super(new URL[] {}, null);
         this.parent = parent;
      }

      @Override
      public void addURL(URL url)
      {
         if (urlCache.contains(url.toExternalForm()) == false)
         {
            log.debug("Add URL: " + url);
            urlCache.add(url.toExternalForm());
            super.addURL(url);
         }
      }

      @Override
      public Class<?> loadClass(String name) throws ClassNotFoundException
      {
         Class<?> clazz;
         try
         {
            clazz = super.loadClass(name);
         }
         catch (ClassNotFoundException ex)
         {
            clazz = parent.loadClass(name);
         }

         return clazz;
      }
   }
}