/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework.plugins;

//$Id: BundleLifecyclePluginImpl.java 92647 2009-08-21 01:18:12Z thomas.diesler@jboss.com $

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.BundleEventsPlugin;
import org.jboss.osgi.jbossmc.api.BundleFactoryPlugin;
import org.jboss.osgi.jbossmc.api.BundleLifecyclePlugin;
import org.jboss.osgi.jbossmc.api.BundleRegistryPlugin;
import org.jboss.osgi.jbossmc.api.BundleResolverPlugin;
import org.jboss.osgi.jbossmc.api.ClassLoaderFactoryPlugin;
import org.jboss.osgi.jbossmc.api.FrameworkEventsPlugin;
import org.jboss.osgi.jbossmc.framework.BundleContextImpl;
import org.jboss.osgi.jbossmc.framework.BundleImpl;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.FrameworkEvent;
import org.osgi.framework.launch.Framework;

/**
 * An implementation of the OSGi bundle lifecycle
 * 
 * @author thomas.diesler@jboss.com
 * @since 29-Jul-2009
 */
public class BundleLifecyclePluginImpl extends AbstractPluginImpl implements BundleLifecyclePlugin
{
   // Provide logging
   final Logger log = Logger.getLogger(BundleLifecyclePluginImpl.class);

   public BundleLifecyclePluginImpl()
   {
   }

   public BundleLifecyclePluginImpl(Framework framework)
   {
      setFramework(framework);
   }

   public Bundle installBundle(String location) throws BundleException
   {
      // Convert location to a virtual file URL
      BundleFactoryPlugin bundleFactory = getPlugin(BundleFactoryPlugin.class);
      location = bundleFactory.getVirtualLocation(location); 
      
      // 1.  If a bundle containing the same location identifier is already installed, the Bundle object for that bundle is returned.
      BundleRegistryPlugin bundleRegistry = getPlugin(BundleRegistryPlugin.class);
      Bundle bundle = bundleRegistry.getBundleByLocation(location);
      if (bundle != null)
         return bundle;
      
      // 2. The bundle's content is read from the input stream. If this fails, a BundleException is thrown.
      // 3. The bundle's associated resources are allocated. The associated resources minimally consist of a unique identifier and a persistent storage area if the platform has file system support. 
      bundle = bundleFactory.createBundle(location);
      bundleRegistry.registerBundle(bundle);

      // Initialize the bundle's runtime class loader
      BundleImpl bundleImpl = (BundleImpl)bundle;
      ClassLoaderFactoryPlugin factory = getPlugin(ClassLoaderFactoryPlugin.class);
      bundleImpl.setRuntimeClassLoader(factory.createClassLoader(bundle));
      
      // 4. The bundle's state is set to INSTALLED.
      bundleImpl.setState(Bundle.INSTALLED);
      
      // 5. A bundle event of type BundleEvent.INSTALLED is fired.
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.INSTALLED, bundle));
      
      // 6. The Bundle object for the newly or previously installed bundle is returned. 
      return bundle;
   }
   
   public void start(Bundle bundle, int options) throws BundleException
   {
      // If this bundle's state is UNINSTALLED then an IllegalStateException is thrown.
      if (bundle.getState() == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already uninstalled: " + bundle);

      // [TODO] Implement the optional Start Level service 

      // [TODO] If this bundle is in the process of being activated or deactivated then this method must wait 
      // for activation or deactivation to complete before continuing.
      if (bundle.getState() == Bundle.STARTING || bundle.getState() == Bundle.STARTING)
         throw new IllegalStateException("Bundle already staring/stopping: " + bundle);

      // If this bundle's state is not RESOLVED, an attempt is made to resolve this bundle.
      if (bundle.getState() != Bundle.RESOLVED)
      {
         BundleResolverPlugin resolver = getPlugin(BundleResolverPlugin.class);
         resolver.resolveBundle(bundle);
         
         BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
         eventManager.fireBundleEvent(new BundleEvent(BundleEvent.RESOLVED, bundle));
      }

      // [TODO] Lazy bundle activation

      // This bundle's state is set to STARTING. 
      setState(bundle, Bundle.STARTING);

      // A bundle event of type BundleEvent.STARTING is fired.
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STARTING, bundle));

      // The BundleActivator is called
      String activatorClass = (String)bundle.getHeaders().get(Constants.BUNDLE_ACTIVATOR);
      if (activatorClass != null)
      {
         try
         {
            BundleActivator activator = (BundleActivator)bundle.loadClass(activatorClass).newInstance();
            activator.start(bundle.getBundleContext());
            
            BundleImpl bundleImpl = (BundleImpl)bundle;
            bundleImpl.setBundleActivator(activator);
         }
         catch (Exception ex)
         {
            // This bundle's state is set to STOPPING.
            setState(bundle, Bundle.STOPPING);

            // A bundle event of type BundleEvent.STOPPING is fired
            eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STOPPING, bundle));

            // Any services registered by this bundle must be unregistered.
            BundleContextImpl contextImpl = (BundleContextImpl)bundle.getBundleContext();
            contextImpl.releaseRegisteredServices();

            // Any services used by this bundle must be released.
            contextImpl.releaseUsedServices();

            // Any listeners registered by this bundle must be removed.
            contextImpl.releaseRegisteredListeners();

            // This bundle's state is set to RESOLVED.
            setState(bundle, Bundle.RESOLVED);

            // A bundle event of type BundleEvent.STOPPED is fired.
            eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STOPPED, bundle));

            // A BundleException is then thrown. 
            throw new BundleException("Cannot call BundleActivator: " + activatorClass, ex);
         }
      }

      // If this bundle's state is UNINSTALLED, because this bundle was uninstalled while 
      // the BundleActivator.start was running 
      if (bundle.getState() == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already uninstalled: " + bundle);

      // This bundle's state is set to ACTIVE
      setState(bundle, Bundle.ACTIVE);

      // A bundle event of type BundleEvent.STARTED is fired
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STARTED, bundle));
   }

   public void stop(Bundle bundle, int options) throws BundleException
   {
      // 1. If this bundle's state is UNINSTALLED then an IllegalStateException is thrown.
      if (bundle.getState() == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already unistalled: " + bundle);
      
      // [TODO] 2. If this bundle is in the process of being activated or deactivated then this method must wait for activation or deactivation to complete
      
      // [TODO] 3. If the STOP_TRANSIENT option is not set then then set this bundle's persistent autostart setting to to Stopped. When the Framework is restarted and this bundle's autostart setting is Stopped, this bundle must not be automatically started.
      
      // 4. If this bundle's state is not STARTING or ACTIVE then this method returns immediately.
      if (!(bundle.getState() == Bundle.STARTING || bundle.getState() == Bundle.ACTIVE))
         return;
         
      // 5. This bundle's state is set to STOPPING.
      int statePriorToStopping = bundle.getState();
      setState(bundle, Bundle.STOPPING);
      
      // 6. A bundle event of type BundleEvent.STOPPING is fired.
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STOPPING, bundle));
      
      // 7. If this bundle's state was ACTIVE prior to setting the state to STOPPING, the BundleActivator.stop(org.osgi.framework.BundleContext) method of this bundle's BundleActivator, if one is specified, is called.
      Exception activatorStopException = null;
      BundleImpl bundleImpl = (BundleImpl)bundle;
      BundleActivator activator = bundleImpl.getBundleActivator();
      if (statePriorToStopping == Bundle.ACTIVE && activator != null)
      {
         try
         {
            activator.stop(bundle.getBundleContext());
         }
         catch (Exception ex)
         {
            // This method must continue to stop this bundle and a BundleException must be thrown after completion of the remaining steps.
            activatorStopException = ex;
         }
      }
      
      // 8. Any services registered by this bundle must be unregistered.
      BundleContextImpl contextImpl = (BundleContextImpl)bundle.getBundleContext();
      contextImpl.releaseRegisteredServices();
      
      // 9. Any services used by this bundle must be released.
      contextImpl.releaseUsedServices();
      
      // 10. Any listeners registered by this bundle must be removed.
      contextImpl.releaseRegisteredListeners();
      
      // 11. If this bundle's state is UNINSTALLED, because this bundle was uninstalled while the BundleActivator.stop method was running, a BundleException must be thrown.
      if (bundle.getState() == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already unistalled: " + bundle);
      
      // 12. This bundle's state is set to RESOLVED.
      setState(bundle, Bundle.RESOLVED);
      
      // 13. A bundle event of type BundleEvent.STOPPED is fired.    
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.STOPPED, bundle));
      
      // A BundleException must be thrown after completion of the remaining steps
      if (activatorStopException != null)
      {
         if (activatorStopException instanceof BundleException)
            throw (BundleException)activatorStopException;
         
         throw new BundleException("Exception in BundleActivator.stop()", activatorStopException);
      }
   }

   public void uninstall(Bundle bundle) throws BundleException
   {
      // 1. If this bundle's state is UNINSTALLED then an IllegalStateException is thrown.
      if (bundle.getState() == Bundle.UNINSTALLED)
         throw new IllegalStateException("Bundle already unistalled: " + bundle);

      // 2. If this bundle's state is ACTIVE, STARTING or STOPPING, this bundle is stopped as described in the Bundle.stop method. 
      if (bundle.getState() == Bundle.ACTIVE || bundle.getState() == Bundle.STARTING || bundle.getState() == Bundle.STOPPING)
      {
         try
         {
            stop(bundle, 0);
         }
         catch (Exception ex)
         {
            // If Bundle.stop throws an exception, a Framework event of type FrameworkEvent.ERROR is fired containing the exception.
            FrameworkEventsPlugin eventManager = getPlugin(FrameworkEventsPlugin.class);
            eventManager.fireFrameworkEvent(new FrameworkEvent(FrameworkEvent.ERROR, bundle, ex));
         }
      }

      // 3. This bundle's state is set to UNINSTALLED.
      setState(bundle, Bundle.UNINSTALLED);

      // 4. A bundle event of type BundleEvent.UNINSTALLED is fired.
      BundleEventsPlugin eventManager = getPlugin(BundleEventsPlugin.class);
      eventManager.fireBundleEvent(new BundleEvent(BundleEvent.UNINSTALLED, bundle));

      // 5. This bundle and any persistent storage area provided for this bundle by the Framework are removed.
      BundleRegistryPlugin bundleRegistry = getPlugin(BundleRegistryPlugin.class);
      bundleRegistry.unregisterBundle(bundle);

      // [TODO] Any persistent storage area provided for this bundle are removed.
   }

   private void setState(Bundle bundle, int state)
   {
      BundleImpl bundleImpl = (BundleImpl)bundle;
      bundleImpl.setState(state);
   }
}