/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework.plugins;

//$Id: $

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.ServiceEventsPlugin;
import org.jboss.osgi.spi.util.ConstantsHelper;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.launch.Framework;

/**
 * A simple implementation of a BundleEventManager
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Aug-2009
 */
public class ServiceEventsPluginImpl extends AbstractPluginImpl implements ServiceEventsPlugin
{
   // Provide logging
   final Logger log = Logger.getLogger(ServiceEventsPluginImpl.class);
   
   private Map<ServiceListener, ListenerRegistration> listeners = new LinkedHashMap<ServiceListener, ListenerRegistration>();
   private boolean enabled;
   
   public ServiceEventsPluginImpl()
   {
   }

   public ServiceEventsPluginImpl(Framework framework)
   {
      setFramework(framework);
   }

   public boolean isEnabled()
   {
      return enabled;
   }

   public void setEnabled(boolean enabled)
   {
      this.enabled = enabled;
   }

   public void fireServiceEvent(ServiceEvent event)
   {
      if (enabled == false)
         return;
      
      String typeStr = ConstantsHelper.serviceEvent(event.getType());
      log.debug("ServiceEvent " + typeStr + " " + event.getSource());
      
      ServiceReference sref = event.getServiceReference();
      String[] clazzes = (String[])sref.getProperty(Constants.OBJECTCLASS);
      if (clazzes == null)
         throw new IllegalStateException("Cannot obtain property '" + Constants.OBJECTCLASS + "' from: " + sref);
      
      // When a ServiceEvent is fired, it is synchronously delivered to a ServiceListener.
      for (ListenerRegistration reg : new ArrayList<ListenerRegistration>(listeners.values()))
      {
         ServiceListener listener = reg.getListener();
         
         // ServiceEvent object delivery to ServiceListener objects is filtered
         Filter filter = reg.getFilter();
         if (filter == null || filter.match(sref))
            listener.serviceChanged(event);
      }
      
      // [TODO] ServiceEvent object delivery to ServiceListener objects is further filtered according to package sources
   }

   public void addServiceListener(ServiceListener listener)
   {
      listeners.put(listener, new ListenerRegistration(listener, null));
   }

   public void addServiceListener(ServiceListener listener, String filter)
   {
      listeners.put(listener, new ListenerRegistration(listener, filter));
   }
   
   public void removeServiceListener(ServiceListener listener)
   {
      listeners.remove(listener);
   }
   
   private static class ListenerRegistration
   {
      ServiceListener listener; 
      Filter filter;
      
      ListenerRegistration(ServiceListener listener, String filterStr) 
      {
         this.listener = listener;
         
         if (filterStr != null)
         {
            try
            {
               filter = FrameworkUtil.createFilter(filterStr);
            }
            catch (InvalidSyntaxException ex)
            {
               throw new IllegalArgumentException(ex);
            }
         }
      }

      public ServiceListener getListener()
      {
         return listener;
      }

      public Filter getFilter()
      {
         return filter;
      }
   }
}