/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossmc.framework.plugins;

//$Id: $

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossmc.api.ServiceEventsPlugin;
import org.jboss.osgi.jbossmc.api.ServiceRegistryPlugin;
import org.jboss.osgi.jbossmc.framework.ServiceReferenceImpl;
import org.jboss.osgi.jbossmc.framework.ServiceRegistrationImpl;
import org.osgi.framework.Bundle;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceFactory;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.launch.Framework;

/**
 * A simple implementation of a BundleRegistry
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Aug-2009
 */
public class ServiceRegistryPluginImpl extends AbstractPluginImpl implements ServiceRegistryPlugin
{
   // Provide logging
   final Logger log = Logger.getLogger(ServiceRegistryPluginImpl.class);
   
   private Map<String, List<ServiceReference>> registry = new HashMap<String, List<ServiceReference>>();
   private long serviceId; 

   public ServiceRegistryPluginImpl()
   {
   }

   public ServiceRegistryPluginImpl(Framework framework)
   {
      setFramework(framework);
   }

   public ServiceRegistration registerService(Bundle bundle, String clazz, Object service, Dictionary<String, Object> properties)
   {
      return registerServiceInternal(bundle, new String[] {clazz}, service, properties);
   }

   public ServiceRegistration registerService(Bundle bundle, String[] clazzes, Object service, Dictionary<String, Object> properties)
   {
      return registerServiceInternal(bundle, clazzes, service, properties);
   }

   public void unregisterService(ServiceRegistration registration)
   {
      // 1. The service is removed from the Framework service registry so that it can no longer be obtained.
      ServiceReference sref = registration.getReference();
      String[] clazzes = (String[])sref.getProperty(Constants.OBJECTCLASS);
      for (String clazz : clazzes)
      {
         List<ServiceReference> srefs = registry.get(clazz);
         if (srefs != null)
         {
            Iterator<ServiceReference> itRefs = srefs.iterator();
            while (itRefs.hasNext())
            {
               ServiceReference next = itRefs.next();
               if (sref == next)
                  itRefs.remove();
            }
         }
      }
      
      // 2. A service event of type ServiceEvent.UNREGISTERING is fired so that bundles using this service can release their use of the service.
      ServiceEventsPlugin serviceEvents = getPlugin(ServiceEventsPlugin.class);
      serviceEvents.fireServiceEvent(new ServiceEvent(ServiceEvent.UNREGISTERING, sref));
      
      // 3. For each bundle whose use count for this service is greater than zero: 
      //    The bundle's use count for this service is set to zero.
      //    If the service was registered with a ServiceFactory object, the ServiceFactory.ungetService method is called to release the service object for the bundle.
      ServiceReferenceImpl srefImpl = (ServiceReferenceImpl)sref;
      srefImpl.unregister();
   }

   public ServiceReference getServiceReference(String clazz)
   {
      ServiceReference srefBest = null;
      
      List<ServiceReference> srefs = registry.get(clazz);
      if (srefs != null && srefs.size() > 0)
      {
         // If multiple such services exist, the service with the highest ranking is returned.
         // If there is a tie in ranking, the service with the lowest service ID (i.e. the service that was registered first) is returned. 
         for (ServiceReference sref : srefs)
         {
            if (srefBest != null)
            {
               Integer bestRanking = (Integer)srefBest.getProperty(Constants.SERVICE_RANKING);
               Long bestId = (Long)srefBest.getProperty(Constants.SERVICE_ID);
               
               Integer thisRanking = (Integer)sref.getProperty(Constants.SERVICE_RANKING);
               Long thisId = (Long)sref.getProperty(Constants.SERVICE_ID);
               if (thisRanking != null)
               {
                  if (bestRanking == null)
                     bestRanking = new Integer(0);
                  
                  if (bestRanking < thisRanking)
                     srefBest = sref;
                  
                  if (bestRanking.equals(thisRanking) && bestId < thisId)
                     srefBest = sref;
               }
            }
            
            if (srefBest == null)
               srefBest = sref;
         }
      }
      
      return srefBest;
   }
   
   public ServiceReference[] getServiceReferences(String clazz, String filterStr) throws InvalidSyntaxException
   {
      Set<ServiceReference> resultRefs = new LinkedHashSet<ServiceReference>();
      
      // If the specified class name, clazz, is not null, the service must have been registered with the specified class name.
      if (clazz != null)
      {
         List<ServiceReference> srefs = registry.get(clazz);
         if (srefs != null)
            resultRefs.addAll(srefs);
      }
      
      // null for all services
      if (clazz == null)
      {
         for (String auxclazz : registry.keySet())
         {
            List<ServiceReference> srefs = registry.get(auxclazz);
            if (srefs != null)
               resultRefs.addAll(srefs);
         }
      }
      
      // If the specified filter is not null, the filter expression must match the service.
      if (filterStr != null)
      {
         Filter filter = FrameworkUtil.createFilter(filterStr);
         Iterator<ServiceReference> itref = resultRefs.iterator();
         while (itref.hasNext())
         {
            ServiceReference sref = itref.next();
            if (filter.match(sref) == false)
               itref.remove();
         }
      }
      
      // [TODO] If the Java Runtime Environment supports permissions, the caller must have ServicePermission with the GET action for at least one of the class names under which the service was registered.
      
      // [TODO] For each class name with which the service was registered, calling ServiceReference.isAssignableTo(Bundle, String) with the context bundle and the class name on the service's ServiceReference object must return true
      
      // Return null if no services are registered which satisfy the search
      if (resultRefs.size() == 0)
         return null;
      
      ServiceReference[] srefArr = new ServiceReference[resultRefs.size()];
      return resultRefs.toArray(srefArr);
   }

   public Object getService(Bundle bundle, ServiceReference sref)
   {
      if (sref == null)
         throw new IllegalArgumentException("Cannot get service for null reference");
      
      ServiceReferenceImpl srefImpl = (ServiceReferenceImpl)sref;
      
      // If the service has been unregistered, null is returned.
      if (srefImpl.isUnregistered())
         return null;
      
      // The context bundle's use count for this service is incremented by one.
      srefImpl.incrementUseCount(bundle);
      
      // The service was registered with an object implementing the ServiceFactory interface
      Object service = srefImpl.getService();
      if (service instanceof ServiceFactory)
      {
         if (srefImpl.getUseCount(bundle) == 1)
         {
            ServiceFactory factory = (ServiceFactory)service;
            service = factory.getService(bundle, srefImpl.getServiceRegistration());
            srefImpl.setCachedService(bundle, service);
         }
         else
         {
            service = srefImpl.getCachedService(bundle);
         }
      }
      
      // The service object for the service is returned.
      return service;
   }
   
   public boolean ungetService(Bundle bundle, ServiceReference sref)
   {
      if (sref == null)
         throw new IllegalArgumentException("Cannot unget service for null reference");
      
      ServiceReferenceImpl srefImpl = (ServiceReferenceImpl)sref;
      
      // 1. If the context bundle's use count for the service is zero or the service has been unregistered, false is returned.
      if (srefImpl.getUseCount(bundle) == 0 || srefImpl.isUnregistered())
         return false;
      
      // 2. The context bundle's use count for this service is decremented by one.
      srefImpl.decrementUseCount(bundle);
      
      // 3. If the context bundle's use count for the service is currently zero and the service was registered with a ServiceFactory object, 
      // the ServiceFactory.ungetService(Bundle, ServiceRegistration, Object) method is called to release the service object for the context bundle.
      Object service = srefImpl.getService();
      if (srefImpl.getUseCount(bundle) == 0 && service instanceof ServiceFactory)
      {
         ServiceFactory factory = (ServiceFactory)service;
         service = srefImpl.getCachedService(bundle);
         factory.ungetService(bundle, srefImpl.getServiceRegistration(), service);
         srefImpl.setCachedService(bundle, null);
      }
      
      // 4. true is returned. 
      return true;
   }

   private ServiceRegistration registerServiceInternal(Bundle bundle, String[] clazzes, Object service, Dictionary<String, Object> props)
   {
      // If service is not a ServiceFactory, an IllegalArgumentException is thrown if service 
      // is not an instanceof all the specified class names.
      if ((service instanceof ServiceFactory) == false)
      {
         for (String clazz : clazzes)
         {
            try
            {
               Class<?> interf = bundle.loadClass(clazz);
               if (interf.isAssignableFrom(service.getClass()) == false)
                  throw new IllegalArgumentException("Service is not assignable to: " + clazz);
            }
            catch (ClassNotFoundException ex)
            {
               throw new IllegalArgumentException("Cannot load service interface: " + clazz, ex);
            }
         }
      }
      
      // 2. The Framework adds the following service properties to the service properties from the specified Dictionary (which may be null):
      //    A property named Constants.SERVICE_ID identifying the registration number of the service
      //    A property named Constants.OBJECTCLASS containing all the specified classes.
      Dictionary<String, Object> properties = new Hashtable<String, Object>();
      properties.put(Constants.SERVICE_ID, new Long(serviceId++));
      properties.put(Constants.OBJECTCLASS, clazzes);
      
      if (props != null)
      {
         Enumeration<String> keys = props.keys();
         while(keys.hasMoreElements())
         {
            String key = keys.nextElement();
            Object value = props.get(key);
            properties.put(key, value);
         }
      }
      
      // 3. The service is added to the Framework service registry and may now be used by other bundles.
      ServiceReferenceImpl sref = new ServiceReferenceImpl(bundle, service, properties);
      for (String clazz : clazzes)
      {
         List<ServiceReference> list = registry.get(clazz);
         if (list == null)
         {
            list = new ArrayList<ServiceReference>();
            registry.put(clazz, list);
         }
         list.add(sref);
      }
      
      // 4. A service event of type ServiceEvent.REGISTERED is fired.
      ServiceEventsPlugin eventManager = getPlugin(ServiceEventsPlugin.class);
      eventManager.fireServiceEvent(new ServiceEvent(ServiceEvent.REGISTERED, sref));
      
      // 5. A ServiceRegistration object for this registration is returned.
      return new ServiceRegistrationImpl(this, sref);
   }
}