/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.osgi.jbosgi151;

import static org.junit.Assert.assertEquals;

import org.jboss.osgi.spi.framework.OSGiBootstrap;
import org.jboss.osgi.spi.framework.OSGiBootstrapProvider;
import org.jboss.osgi.spi.testing.OSGiTestHelper;
import org.jboss.test.osgi.jbosgi151.bundleA.BeanA;
import org.jboss.test.osgi.jbosgi151.bundleB.BeanB;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.launch.Framework;

/**
 * [JBOSGI-151] Cannot resolve circular dependencies
 * 
 * https://jira.jboss.org/jira/browse/JBOSGI-151
 * 
 * BundleA 
 *  exports A
 *  imports B
 *  
 * BundleB 
 *  exports B
 *  imports A
 *  
 * BundleC 
 *  exports A, B
 *  imports A
 * 
 * BundleD 
 *  exports A,
 *  imports A, B
 * 
 * @author thomas.diesler@jboss.com
 * @since 07-Sep-2009
 */
public class OSGI151TestCase
{
   private static Framework framework;
   
   @BeforeClass
   public static void beforeClass() throws Exception
   {
      OSGiBootstrapProvider bootProvider = OSGiBootstrap.getBootstrapProvider();
      framework = bootProvider.getFramework();
      framework.start();
   }
   
   @BeforeClass
   public static void afterClass() throws Exception
   {
      if (framework != null)
      {
         framework.stop();
         framework.waitForStop(2000);
      }
   }
   
   @Test
   @Ignore
   public void testCircularNoSelfDependency() throws Exception
   {
      BundleContext sysContext = framework.getBundleContext();
      
      Bundle bundleA = sysContext.installBundle(getBundleLocation("jbosgi151-bundleA.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleA.getState());

      Bundle bundleB = sysContext.installBundle(getBundleLocation("jbosgi151-bundleB.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleB.getState());
      
      bundleB.start();
      assertEquals("ACTIVE expected", Bundle.ACTIVE, bundleB.getState());
      assertEquals("RESOLVED expected", Bundle.RESOLVED, bundleA.getState());
      
      Class<?> classAA = bundleA.loadClass(BeanA.class.getName());
      Class<?> classAB = bundleB.loadClass(BeanA.class.getName());
      assertEquals("Class for BeanA", classAA, classAB);
      
      Class<?> classBA = bundleA.loadClass(BeanB.class.getName());
      Class<?> classBB = bundleB.loadClass(BeanB.class.getName());
      assertEquals("Class for BeanB", classBA, classBB);
      
      bundleB.uninstall();
      bundleA.uninstall();
   }

   @Test
   public void testCircularInstallCbeforeD() throws Exception
   {
      BundleContext sysContext = framework.getBundleContext();
      
      Bundle bundleC = sysContext.installBundle(getBundleLocation("jbosgi151-bundleC.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleC.getState());

      Bundle bundleD = sysContext.installBundle(getBundleLocation("jbosgi151-bundleD.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleD.getState());
      
      bundleD.start();
      assertEquals("ACTIVE expected", Bundle.ACTIVE, bundleD.getState());
      assertEquals("RESOLVED expected", Bundle.RESOLVED, bundleC.getState());
      
      Class<?> classBC = bundleC.loadClass(BeanB.class.getName());
      Class<?> classBD = bundleD.loadClass(BeanB.class.getName());
      assertEquals("Class for BeanB", classBC, classBD);
      
      Class<?> classAC = bundleC.loadClass(BeanA.class.getName());
      Class<?> classAD = bundleD.loadClass(BeanA.class.getName());
      assertEquals("Class for BeanA", classAC, classAD);
      
      bundleD.uninstall();
      bundleC.uninstall();
   }

   @Test
   public void testCircularInstallDbeforeC() throws Exception
   {
      BundleContext sysContext = framework.getBundleContext();
      
      Bundle bundleD = sysContext.installBundle(getBundleLocation("jbosgi151-bundleD.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleD.getState());
      
      Bundle bundleC = sysContext.installBundle(getBundleLocation("jbosgi151-bundleC.jar"));
      assertEquals("INSTALLED expected", Bundle.INSTALLED, bundleC.getState());

      bundleD.start();
      assertEquals("ACTIVE expected", Bundle.ACTIVE, bundleD.getState());
      assertEquals("RESOLVED expected", Bundle.RESOLVED, bundleC.getState());
      
      Class<?> classBC = bundleC.loadClass(BeanB.class.getName());
      Class<?> classBD = bundleD.loadClass(BeanB.class.getName());
      assertEquals("Class for BeanB", classBC, classBD);
      
      Class<?> classAC = bundleC.loadClass(BeanA.class.getName());
      Class<?> classAD = bundleD.loadClass(BeanA.class.getName());
      assertEquals("Class for BeanA", classAC, classAD);
      
      bundleD.uninstall();
      bundleC.uninstall();
   }
   
   private String getBundleLocation(String jarname)
   {
      return new OSGiTestHelper().getTestArchiveURL(jarname).toExternalForm();
   }
}