/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.osgi.service.microcontainer;

//$Id: MicrocontainerServiceTestCase.java 93518 2009-09-15 09:54:24Z thomas.diesler@jboss.com $

import static org.jboss.osgi.spi.management.ManagedFrameworkMBean.MBEAN_MANAGED_FRAMEWORK;
import static org.jboss.osgi.spi.management.MicrocontainerServiceMBean.MBEAN_MICROCONTAINER_SERVICE;
import static org.jboss.osgi.spi.service.DeployerService.MBEAN_DEPLOYER_SERVICE;
import static org.jboss.test.osgi.service.microcontainer.bundleB.SomeBeanMBean.MBEAN_NAME;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.osgi.jbossxb.XMLBindingCapability;
import org.jboss.osgi.jmx.JMXCapability;
import org.jboss.osgi.jndi.JNDICapability;
import org.jboss.osgi.spi.capability.MicrocontainerCapability;
import org.jboss.osgi.spi.management.MBeanProxy;
import org.jboss.osgi.spi.management.ManagedFrameworkMBean;
import org.jboss.osgi.spi.management.MicrocontainerServiceMBean;
import org.jboss.osgi.spi.service.DeployerService;
import org.jboss.osgi.spi.service.MicrocontainerService;
import org.jboss.osgi.spi.testing.OSGiRuntime;
import org.jboss.osgi.spi.testing.OSGiTest;
import org.jboss.osgi.spi.testing.OSGiTestHelper;
import org.jboss.test.osgi.service.microcontainer.bundleB.SomeBeanMBean;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * A test that checks {@link MicrocontainerService} functionality
 * 
 * @author thomas.diesler@jboss.com
 * @since 12-Feb-2009
 */
public class MicrocontainerServiceTestCase extends OSGiTest
{
   private static OSGiRuntime runtime;

   @BeforeClass
   public static void setUpClass() throws Exception
   {
      runtime = new OSGiTestHelper().getDefaultRuntime();
      
      runtime.addCapability(new JNDICapability());
      runtime.addCapability(new JMXCapability());
      runtime.addCapability(new XMLBindingCapability());
      runtime.addCapability(new MicrocontainerCapability());
      
   }

   @AfterClass
   public static void tearDownClass() throws Exception
   {
      runtime.shutdown();
   }

   @Test
   public void testServiceAccess() throws Exception
   {
      MicrocontainerServiceMBean mcService = MBeanProxy.get(MicrocontainerServiceMBean.class, MBEAN_MICROCONTAINER_SERVICE, runtime.getMBeanServer());
      List<String> registeredBeans = mcService.getRegisteredBeans();
      assertTrue("BundleContext registered with MC", registeredBeans.contains(MicrocontainerService.BEAN_SYSTEM_BUNDLE_CONTEXT));
      assertTrue("MBeanServer registered with MC", registeredBeans.contains(MicrocontainerService.BEAN_MBEAN_SERVER));
      assertTrue("Kernel registered with MC", registeredBeans.contains(MicrocontainerService.BEAN_KERNEL));
   }

   /**
    * Test whether a bundle can be deployed through the {@link DeployerService}
    */
   @Test
   public void testBundleDeployment() throws Exception
   {
      DeployerService deployer = MBeanProxy.get(DeployerService.class, MBEAN_DEPLOYER_SERVICE, runtime.getMBeanServer());
      deployer.deploy(getTestArchiveURL("service/mcservice-bundleA.jar"));

      ManagedFrameworkMBean frameworkMBean = MBeanProxy.get(ManagedFrameworkMBean.class, MBEAN_MANAGED_FRAMEWORK, runtime.getMBeanServer());
      Set<ObjectName> bundles = frameworkMBean.getBundles();
      assertTrue("Managed bundle registered", bundles.toString().indexOf("bundle=mcservice-bundleA") > 0);

      deployer.undeploy(getTestArchiveURL("service/mcservice-bundleA.jar"));
   }

   /**
    * Test whether a bundle that contains beans can be deployed through the {@link DeployerService}
    */
   @Test
   public void testBeansDeployment() throws Exception
   {
      MicrocontainerServiceMBean mcService = MBeanProxy.get(MicrocontainerServiceMBean.class, MBEAN_MICROCONTAINER_SERVICE, runtime.getMBeanServer());
      DeployerService deployer = MBeanProxy.get(DeployerService.class, MBEAN_DEPLOYER_SERVICE, runtime.getMBeanServer());
      
      deployer.deploy(getTestArchiveURL("service/mcservice-bundleB.jar"));

      ManagedFrameworkMBean frameworkMBean = MBeanProxy.get(ManagedFrameworkMBean.class, MBEAN_MANAGED_FRAMEWORK, runtime.getMBeanServer());
      Set<ObjectName> bundles = frameworkMBean.getBundles();
      assertTrue("Managed bundle registered", bundles.toString().indexOf("bundle=mcservice-bundleB") > 0);

      // Check whether the bean is registered
      List<String> registeredBeans = mcService.getRegisteredBeans();
      assertTrue("SomeBean registered", registeredBeans.contains("SomeBean"));

      // Access the bean through JMX
      SomeBeanMBean someBean = MBeanProxy.get(SomeBeanMBean.class, MBEAN_NAME, runtime.getMBeanServer());
      assertEquals("hello", someBean.echo("hello"));
      
      deployer.undeploy(getTestArchiveURL("service/mcservice-bundleB.jar"));

      // Check whether the bean is unregistered
      registeredBeans = mcService.getRegisteredBeans();
      assertFalse("SomeBean unregistered", registeredBeans.contains("SomeBean"));
   }
}