/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.reflect;

// $Id: BlueprintMetadata.java 91345 2009-07-16 14:16:10Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.parser.xb.TActivation;
import org.jboss.osgi.blueprint.parser.xb.TBlueprint;
import org.jboss.osgi.blueprint.parser.xb.TComponent;
import org.jboss.osgi.spi.NotImplementedException;
import org.osgi.service.blueprint.container.NoSuchComponentException;
import org.osgi.service.blueprint.reflect.ComponentMetadata;

/**
 * The blueprint element is the top element.
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
public class BlueprintMetadata extends MetadataImpl
{
   private TBlueprint tBlueprint;
   private Map<String, ComponentMetadata> components = new LinkedHashMap<String, ComponentMetadata>();

   public BlueprintMetadata(BlueprintContext context, TBlueprint tBlueprint)
   {
      super(context, null);
      this.tBlueprint = tBlueprint;
   }
   
   public int getDefaultActivation()
   {
      TActivation activation = tBlueprint.getDefaultActivation();
      if (activation == TActivation.lazy)
         return ComponentMetadata.ACTIVATION_LAZY;
      else if (activation == TActivation.eager)
         return ComponentMetadata.ACTIVATION_EAGER;
      else
         throw new IllegalStateException("Unsuported activation code: " + activation);
   }

   public void initialize()
   {
      ComponentFactory factory = new ComponentFactory(context, this);
      for (TComponent tComp : tBlueprint.getComponents())
      {
         ComponentMetadata comp = factory.createComponent(tComp);
         String key = comp.getId();
         
         // All top level components share a single namespace.
         // That is, it is an error if the same id is used multiple times.
         if (components.get(key) != null)
            throw new IllegalStateException("Duplicate top-level component: " + key);
         
         components.put(key, comp);
      }
   }

   public ComponentMetadata getComponent(String name)
   {
      ComponentMetadata comp = components.get(name);
      if (comp == null)
         throw new NoSuchComponentException(name);
      
      return comp;
   }

   public List<ComponentMetadata> getComponents()
   {
      ArrayList<ComponentMetadata> list = new ArrayList<ComponentMetadata>(components.values());
      return Collections.unmodifiableList(list);
   }

   // Merge this Blueprint meta data with the given other, which may be null.
   public BlueprintMetadata merge(BlueprintMetadata other)
   {
      if (other == null)
         return this;

      throw new NotImplementedException("Blueprint metadata merge not yet implemented");
   }
}
