/*
 * JBoss, Home of Professional Open Source
 *
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @author tags. See the COPYRIGHT.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.teiid.netty.handler.codec.serialization;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.io.StreamCorruptedException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.teiid.jdbc.JDBCPlugin;

/**
 * @author The Netty Project (netty-dev@lists.jboss.org)
 * @author Trustin Lee (tlee@redhat.com)
 *
 * @version $Rev: 381 $, $Date: 2008-10-01 06:06:18 -0500 (Wed, 01 Oct 2008) $
 *
 */
public class CompactObjectInputStream extends ObjectInputStream {

    private final ClassLoader classLoader;

    CompactObjectInputStream(InputStream in) throws IOException {
        this(in, null);
    }

    public CompactObjectInputStream(InputStream in, ClassLoader classLoader) throws IOException {
        super(in);
        this.classLoader = classLoader;
    }

    @Override
    protected void readStreamHeader() throws IOException,
            StreamCorruptedException {
        int version = readByte() & 0xFF;
        if (version != STREAM_VERSION) {
            throw new StreamCorruptedException(
                    "Unsupported version: " + version); //$NON-NLS-1$
        }
    }

    @Override
    protected ObjectStreamClass readClassDescriptor()
            throws IOException, ClassNotFoundException {
        int type = read();
        if (type < 0) {
            throw new EOFException();
        }
        switch (type) {
        case CompactObjectOutputStream.TYPE_PRIMITIVE:
            return super.readClassDescriptor();
        case CompactObjectOutputStream.TYPE_NON_PRIMITIVE:
            String className = readUTF();
            Class<?> clazz;
            if (classLoader == null) {
                clazz = Class.forName(
                        className, true,
                        CompactObjectInputStream.class.getClassLoader());
            } else {
                clazz = Class.forName(className, true, classLoader);
            }
            return ObjectStreamClass.lookupAny(clazz);
        default:
        	clazz = CompactObjectOutputStream.KNOWN_CODES.get(type);
        	if (clazz != null) {
                return ObjectStreamClass.lookupAny(clazz);
        	}
            throw new StreamCorruptedException(
                    "Unexpected class descriptor type: " + type); //$NON-NLS-1$
        }
    }

    @Override
    protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
        String name = desc.getName();
        
        //deny the resolving of classes that can cause security issues when deserialized
        if (name.endsWith("functors.InvokerTransformer") //$NON-NLS-1$
        		|| name.endsWith("functors.InstantiateTransformer") //$NON-NLS-1$
        		|| name.equals("org.​apache.​commons.​collections.​Transformer") //$NON-NLS-1$
        		|| name.equals("org.codehaus.groovy.runtime.ConvertedClosure") //$NON-NLS-1$
        		|| name.equals("org.codehaus.groovy.runtime.MethodClosure") //$NON-NLS-1$
        		|| name.equals("org.springframework.beans.factory.ObjectFactory") //$NON-NLS-1$
        		|| name.endsWith(".​trax.​TemplatesImpl")) { //$NON-NLS-1$
        	Logger.getLogger("org.teiid").log(Level.SEVERE, JDBCPlugin.Util.gs(JDBCPlugin.Event.TEIID20037, name)); //$NON-NLS-1$
        	throw new ClassNotFoundException(name);
        }
        
        try {
            return Class.forName(name, false, classLoader);
        } catch (ClassNotFoundException ex) {
            return super.resolveClass(desc);
        }
    }
}
