package org.jboss.webbeans.tck.impl.util;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;

/**
 * Utility class to load deployment properties
 * 
 * @author Pete Muir
 */
public class DeploymentProperties
{
   // The resource bundle used to control Web Beans RI deployment
   public static final String RESOURCE_BUNDLE = "META-INF/web-beans-tck.properties";
   
   // The class to work from
   private SimpleResourceLoader resourceLoader;

   /**
    * Constructor
    * 
    * @param classLoader The classloader to work on
    */
   public DeploymentProperties()
   {
      this.resourceLoader = new SimpleResourceLoader();
   }

   /**
    * Get a list of possible values for a given key.
    * 
    * First, System properties are tried, followed by the specified resource
    * bundle (first in classpath only).
    * 
    * @param key The key to search for
    * @return A list of possible values. An empty list is returned if there are
    *         no matches.
    */
   public List<String> getPropertyValues(String key)
   {
      List<String> values = new ArrayList<String>();
      addPropertiesFromSystem(key, values);
      addPropertiesFromResourceBundle(key, values);
      return values;
   }

   /**
    * Adds matches from system properties
    * 
    * @param key The key to match
    * @param values The currently found values
    */
   private void addPropertiesFromSystem(String key, List<String> values)
   {
      addProperty(key, System.getProperty(key), values);
   }

   /**
    * Adds matches from detected resource bundles
    * 
    * @param key The key to match
    * @param values The currently found values
    */
   private void addPropertiesFromResourceBundle(String key, List<String> values)
   {
      try
      {
         for (URL url : resourceLoader.getResources(RESOURCE_BUNDLE))
         {
            Properties properties = new Properties();
            InputStream propertyStream = url.openStream();
            try
            {
               properties.load(propertyStream);
               addProperty(key, properties.getProperty(key), values);
            }
            finally
            {
               if (propertyStream != null)
               {
                  propertyStream.close();
               }
            }
         }
      }
      catch (IOException e)
      {
         // No - op, file is optional
      }
   }

   /**
    * Add the property to the set of properties only if it hasn't already been
    * added
    * 
    * @param key The key searched for
    * @param value The value of the property
    * @param values The currently found values
    */
   private void addProperty(String key, String value, List<String> values)
   {
      if (value != null)
      {
         String[] properties = Strings.split(value, ":");
         for (String property : properties)
         {
            values.add(property);
         }

      }
   }
   
   /**
    * Gets the possible implementation class for a given property for which the
    * values are classanames
    * 
    * @param deploymentProperties The deployment properties object to use
    * @param resourceLoader The resource laoder to use to attempt
    * @param propertyName The name of the property to load
    * @return A set of classes specified
    */
   @SuppressWarnings("unchecked")
   public <T> Set<Class<? extends T>> getClasses(String propertyName, Class<T> expectedType)
   {
      Set<Class<? extends T>> classes = new HashSet<Class<? extends T>>();
      for (String className : getPropertyValues(propertyName))
      {
         try
         {
            classes.add((Class<? extends T>) resourceLoader.classForName(className));
         }
         catch (ResourceLoadingException e)
         {
            //log.debug("Unable to load class " + className + " for property " + propertyName, e);
         }
      }
      return classes;
   }

}