package org.jboss.webbeans.tck.unit.event;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.context.Dependent;
import javax.event.Event;
import javax.event.Observer;
import javax.event.ObserverException;
import javax.inject.DefinitionException;
import javax.inject.DuplicateBindingTypeException;
import javax.inject.Standard;
import javax.inject.TypeLiteral;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.testng.annotations.Test;

/**
 * Event bus tests
 * 
 * @author Nicklas Karlsson
 * @author David Allen
 * 
 *         Spec version: PRD2
 */
public class EventTest extends AbstractTest
{

   @Override
   protected List<Class<? extends Annotation>> getEnabledDeploymentTypes()
   {
      List<Class<? extends Annotation>> deploymentTypes = new ArrayList<Class<? extends Annotation>>();
      deploymentTypes.addAll(getStandardDeploymentTypes());
      deploymentTypes.add(AnotherDeploymentType.class);
      return deploymentTypes;
   }

   public static class AnEventType
   {
   }

   public static class ATemplatedEventType<T>
   {
   }

   public static class AnObserver implements Observer<AnEventType>
   {
      public boolean wasNotified = false;

      public void notify(AnEventType event)
      {
         wasNotified = true;
      }
   }

   public static class AListObserver implements Observer<ArrayList<String>>
   {
      public boolean wasNotified = false;

      public void notify(ArrayList<String> event)
      {
         wasNotified = true;
      }
   }

   public static class AnotherListObserver implements Observer<ArrayList<Integer>>
   {
      public boolean wasNotified = false;

      public void notify(ArrayList<Integer> event)
      {
         wasNotified = true;
      }
   }

   public static class AnObserverWithException implements Observer<AnEventType>
   {
      public boolean wasNotified = false;
      public RuntimeException theException = new RuntimeException("RE1");

      public void notify(AnEventType event)
      {
         wasNotified = true;
         throw theException;
      }
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.1")
   public void testEventTypeIncludesAllSuperclassesAndInterfacesOfEventObject()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.2")
   public void testManagerFireEvent()
   {
      // First a simple event with no bindings is fired
      AnEventType anEvent = new AnEventType();
      manager.fireEvent(anEvent);

      // Next an event with some event bindings is fired
      manager.fireEvent(anEvent, new RoleBinding("Admin"));
   }

   /**
    * If the type of the event object passed to fireEvent() contains type
    * variables or wildcards, an IllegalArgumentException is thrown
    */
   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = { "7.1", "7.2" })
   public void testManagerFireEventWithEventTypeParametersFails()
   {
      ATemplatedEventType<String> anEvent = new ATemplatedEventType<String>();
      manager.fireEvent(anEvent);
   }

   /**
    * If the type of the event object passed to fireEvent() contains type
    * variables or wildcards, an IllegalArgumentException is thrown
    */
   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = { "7.1", "7.2" })
   public void testManagerFireEventWithEventTypeWildcardsFails()
   {
      // Although the above test is really the same as with a wildcard, we will
      // test
      // it anyhow since the specification calls it out separately.
      ATemplatedEventType<?> anEventOnAnyType = new ATemplatedEventType<String>();
      manager.fireEvent(anEventOnAnyType);
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = { "7.1", "7.2" })
   public void testManagerFireEventWithNonBindingAnnotationsFails()
   {
      // The specs are not exactly clear on what is supposed to happen here,
      // but borrowing from Section 8.3, we'll expect the same behavior here
      // for a consistent API.
      // TODO Verify that fireEvent should fail on non-binding annotations
      AnEventType anEvent = new AnEventType();
      manager.fireEvent(anEvent, new AnimalStereotypeAnnotationLiteral());
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.3")
   public void testManagerAddObserver()
   {
      Observer<AnEventType> observer = new AnObserver();

      // First test with the Class<T> of the event type
      manager.addObserver(observer, AnEventType.class);
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType());
      assert resolvedObservers.size() == 1;
      assert resolvedObservers.iterator().next() == observer;

      // Now test with the TypeLiteral<T> of the event type
      observer = new AnObserver();
      manager.addObserver(observer, new TypeLiteral<AnEventType>()
      {
      });
      resolvedObservers = manager.resolveObservers(new AnEventType());
      assert resolvedObservers.size() == 2;
      boolean foundObserver = false;
      for (Observer<AnEventType> obs : resolvedObservers)
      {
         if (obs == observer)
         {
            foundObserver = true;
            break;
         }
      }
      assert foundObserver;

      // Try adding an observer with some binding types
      observer = new AnObserver();
      Annotation[] bindingTypes = new Annotation[] { new RoleBinding("Admin"), new RoleBinding("Manager") };
      manager.addObserver(observer, AnEventType.class, bindingTypes);
      resolvedObservers = manager.resolveObservers(new AnEventType(), bindingTypes);
      assert resolvedObservers.size() == 3;
      foundObserver = false;
      for (Observer<AnEventType> obs : resolvedObservers)
      {
         if (obs == observer)
         {
            foundObserver = true;
            break;
         }
      }
      assert foundObserver;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.3")
   public void testManagerRemoveObserver()
   {
      Observer<AnEventType> observer = new AnObserver();

      // First test with the Class<T> of the event type
      manager.addObserver(observer, AnEventType.class);
      manager.removeObserver(observer, AnEventType.class);
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType());
      assert resolvedObservers.isEmpty();

      // Now test with the TypeLiteral<T> of the event type
      observer = new AnObserver();
      manager.addObserver(observer, new TypeLiteral<AnEventType>()
      {
      });
      manager.removeObserver(observer, new TypeLiteral<AnEventType>()
      {
      });
      resolvedObservers = manager.resolveObservers(new AnEventType());
      assert resolvedObservers.isEmpty();

      // Also test with binding types
      Annotation[] bindings = new Annotation[] { new RoleBinding("Admin") };
      manager.addObserver(observer, AnEventType.class, bindings);
      manager.removeObserver(observer, AnEventType.class);
      resolvedObservers = manager.resolveObservers(new AnEventType(), bindings);
      assert !resolvedObservers.isEmpty();
      manager.removeObserver(observer, AnEventType.class, new RoleBinding("Admin"));
      resolvedObservers = manager.resolveObservers(new AnEventType(), bindings);
      assert resolvedObservers.isEmpty();
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.3")
   public void testMultipleInstancesOfSameBindingTypeWhenAddingObserverFails()
   {
      Observer<AnEventType> observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class, new RoleBinding("Admin"), new TameAnnotationLiteral(), new TameAnnotationLiteral());
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.3")
   public void testNonBindingTypePassedToAddObserverFails()
   {
      Observer<AnEventType> observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class, new AnimalStereotypeAnnotationLiteral());
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.3")
   public void testMultipleInstancesOfSameBindingTypeWhenRemovingObserverFails()
   {
      Observer<AnEventType> observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class);
      manager.removeObserver(observer, AnEventType.class, new RoleBinding("Admin"), new TameAnnotationLiteral(), new TameAnnotationLiteral());
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.3")
   public void testNonBindingTypePassedToRemoveObserverFails()
   {
      Observer<AnEventType> observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class);
      manager.removeObserver(observer, AnEventType.class, new AnimalStereotypeAnnotationLiteral());
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = { "7.1", "7.4" })
   public void testConsumerNotifiedWhenEventTypeAndAllBindingsMatch()
   {
      AnObserver observer1 = new AnObserver();
      AnObserver observer2 = new AnObserver();
      manager.addObserver(observer1, AnEventType.class);
      manager.addObserver(observer2, AnEventType.class);

      // Fire an event that will be delivered to the two above observers
      AnEventType anEvent = new AnEventType();
      manager.fireEvent(anEvent);

      assert observer1.wasNotified;
      assert observer2.wasNotified;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.4")
   public void testObserverThrowsExceptionAbortsNotifications()
   {
      AnObserverWithException observer = new AnObserverWithException();
      AnObserverWithException anotherObserver = new AnObserverWithException();
      manager.addObserver(anotherObserver, AnEventType.class);
      manager.addObserver(observer, AnEventType.class);

      // Fire an event that will be delivered to the two above observers
      AnEventType anEvent = new AnEventType();
      boolean fireFailed = false;
      try
      {
         manager.fireEvent(anEvent);
      }
      catch (Exception e)
      {
         if (e.equals(observer.theException) || e.equals(anotherObserver.theException))
            fireFailed = true;
      }
      assert fireFailed;
      // Only one of the observers can be notified if processing
      // is aborted
      assert observer.wasNotified ^ anotherObserver.wasNotified;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5")
   public void testMultipleObserverMethodsOK()
   {
      // This bean has a couple observer methods
      deployBeans(TeaCupPomeranian.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans.size() >= 2;

      // Resolve the observers for String and Integer
      Set<Observer<String>> resolvedStringObservers = manager.resolveObservers("A new event");
      assert resolvedStringObservers.size() == 1;

      Set<Observer<Integer>> resolvedIntegerObservers = manager.resolveObservers(new Integer(42));
      assert resolvedIntegerObservers.size() == 1;
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = { "7.5.1", "7.5.2" })
   public void testObserverMethodMustHaveOnlyOneEventParameter()
   {
      deployBeans(YorkshireTerrier_Broken.class);

      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.5.1")
   public void testObserverMethodCannotObserveParameterizedEvents()
   {
      deployBeans(BostonTerrier.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.1")
   public void testObserverMethodWithoutBindingTypesObservesEventsWithoutBindingTypes()
   {
      // This observer has no binding types specified
      deployBeans(Pomeranian.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans.size() >= 2;

      // Resolve registered observers with an event containing no binding types
      Set<Observer<String>> resolvedObservers = manager.resolveObservers("A new event");
      assert resolvedObservers.size() == 2;
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.5.2")
   public void testObserverMethodAnnotatedProducesFails()
   {
      deployBeans(BorderTerrier.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.5.2")
   public void testObserverMethodAnnotatedInitializerFails()
   {
      deployBeans(AustralianTerrier.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.5.2")
   public void testObserverMethodWithDisposesParamFails()
   {
      deployBeans(FoxTerrier.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.2")
   public void testObserverMethodMayHaveMultipleBindingTypes()
   {
      deployBeans(BullTerrier.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
      // If we can resolve the observer with the two binding types,
      // then it worked
      Set<Observer<String>> resolvedObservers = manager.resolveObservers("An event object", new RoleBinding("Admin"), new TameAnnotationLiteral());
      assert resolvedObservers.size() == 1;

   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.3")
   public void testXMLDefinedObserverMethodIgnoresBindingAnnotations()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.3")
   public void testXMLDefinedObserverNotFindingImplementationMethodFails()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.4")
   public void testObserverMethodReceivesInjectionsOnNonObservesParameters()
   {
      deployBeans(BananaSpider.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans != null;
   }

   /**
    * Tests that a conditional observer is not notified of events until after it
    * is created by some other separate action.
    */
   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.5")
   public void testConditionalObserver()
   {
      RecluseSpider.notified = false;
      deployBeans(RecluseSpider.class);

      manager.fireEvent("New string event");
      // Should not be notified since bean is not instantiated yet
      assert !RecluseSpider.notified;

      // Now instantiate the bean and fire another event
      try
      {
         activateDependentContext();
         RecluseSpider bean = manager.getInstanceByType(RecluseSpider.class);
         assert bean != null;
         bean.toString();

         manager.fireEvent("Another event");
         assert RecluseSpider.notified;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.1")
   public void testTransactionalObserverCanOnlyObserveSinglePhase()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.6")
   public void testTransactionalObserverNotifiedImmediatelyWhenNoTransactionInProgress()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.6")
   public void testAfterTransactionCompletionObserver()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.6")
   public void testAfterTransactionSuccessObserver()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.6")
   public void testAfterTransactionFailureObserver()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.6")
   public void testBeforeTransactionCompletionObserver()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.8")
   public void testObserverMethodRegistration()
   {
      // This bean has two observer methods, one static and one non-static
      deployBeans(Pomeranian.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert beans.size() >= 2;

      // Resolve registered observers with an event containing no binding types
      Set<Observer<String>> resolvedObservers = manager.resolveObservers("A new event");
      assert resolvedObservers.size() == 2;
   }

   @Test(groups = { "events" }, expectedExceptions = { TeaCupPomeranian.OversizedException.class })
   @SpecAssertion(section = "7.5.8")
   public void testNonTransactionalObserverThrownNonCheckedExceptionIsRethrown()
   {
      deployBeans(TeaCupPomeranian.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert !beans.isEmpty();
      manager.fireEvent("Another event");
   }

   @Test(groups = { "events" }, expectedExceptions = { ObserverException.class })
   @SpecAssertion(section = "7.5.8")
   public void testNonTransactionalObserverThrownCheckedExceptionIsWrappedAndRethrown()
   {
      deployBeans(TeaCupPomeranian.class);
      Set<Bean<Object>> beans = manager.resolveByType(Object.class);
      assert !beans.isEmpty();
      manager.fireEvent(new Integer(1));
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.6")
   public void testDuplicateBindingsToFireFails()
   {
      deployBeans(SweeWaxbill_Broken.class);
      try
      {
         activateDependentContext();
         SweeWaxbill_Broken bean = manager.getInstanceByType(SweeWaxbill_Broken.class);
         bean.methodThatFiresEvent();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.6")
   public void testDuplicateBindingsToObservesFails()
   {
      deployBeans(SweeWaxbill_Broken.class);
      try
      {
         activateDependentContext();
         SweeWaxbill_Broken bean = manager.getInstanceByType(SweeWaxbill_Broken.class);
         bean.methodThatRegistersObserver();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.6")
   public void testNonBindingTypePassedToFireFails()
   {
      deployBeans(OwlFinch_Broken.class);
      try
      {
         activateDependentContext();
         OwlFinch_Broken bean = manager.getInstanceByType(OwlFinch_Broken.class);
         bean.methodThatFiresEvent();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.6")
   public void testNonBindingTypePassedToObservesFails()
   {
      deployBeans(OwlFinch_Broken.class);
      try
      {
         activateDependentContext();
         OwlFinch_Broken bean = manager.getInstanceByType(OwlFinch_Broken.class);
         bean.methodThatRegistersObserver();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6")
   public void testFiresAnnotationOnEventTypes()
   {
      deployBeans(BlueFacedParrotFinch.class);
      try
      {
         activateDependentContext();
         BlueFacedParrotFinch bean = manager.getInstanceByType(BlueFacedParrotFinch.class);
         bean.methodThatRegistersObserver();

         Set<Observer<String>> observers = manager.resolveObservers("String type event");
         assert observers.size() == 1;
      }
      finally
      {
         deactivateDependentContext();
      }
      deployBeans(StarFinch.class, FinchKeeper.class, BirdCage.class);
      try
      {
         activateDependentContext();
         StarFinch starFinch = manager.getInstanceByType(StarFinch.class);
         FinchKeeper birdKeeper = manager.getInstanceByType(FinchKeeper.class);
         BirdCage birdCage = manager.getInstanceByType(BirdCage.class);
         assert starFinch != null;
         assert birdCage != null;
         assert birdCage.getSomeMess() != null;
         assert birdKeeper.isNewMessDetected();
      }
      finally
      {
         deactivateDependentContext();
      }
      deployBeans(OrangeCheekedWaxbill.class, FinchKeeper.class);
      try
      {
         activateDependentContext();
         OrangeCheekedWaxbill bird = manager.getInstanceByType(OrangeCheekedWaxbill.class);
         FinchKeeper birdKeeper = manager.getInstanceByType(FinchKeeper.class);
         assert bird != null;
         assert bird.getSomeMess() != null;
         assert birdKeeper.isNewMessDetected();
      }
      finally
      {
         deactivateDependentContext();
      }
      deployBeans(AuroraFinch.class, FinchKeeper.class);
      try
      {
         activateDependentContext();
         AuroraFinch bird = manager.getInstanceByType(AuroraFinch.class);
         FinchKeeper birdKeeper = manager.getInstanceByType(FinchKeeper.class);
         assert bird != null;
         assert birdKeeper.isNewMessDetected();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.6")
   public void testFiresAnnotationOnNonEventTypeInjectionPointFails()
   {
      deployBeans(CommonWaxbill_Broken.class);
      try
      {
         activateDependentContext();
         CommonWaxbill_Broken bean = manager.getInstanceByType(CommonWaxbill_Broken.class);
         assert bean != null;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.6")
   public void testFiresAnnotationOnInjectionPointWithoutTypeParameterFails()
   {
      deployBeans(BlackRumpedWaxbill_Broken.class);
      try
      {
         activateDependentContext();
         BlackRumpedWaxbill_Broken bean = manager.getInstanceByType(BlackRumpedWaxbill_Broken.class);
         assert bean != null;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.6")
   public void testFiresAnnotationOnInjectionPointWithWildcardedTypeParameterFails()
   {
      deployBeans(GoldbreastWaxbill_Broken.class);
      try
      {
         activateDependentContext();
         GoldbreastWaxbill_Broken bean = manager.getInstanceByType(GoldbreastWaxbill_Broken.class);
         assert bean != null;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" }, expectedExceptions = { DefinitionException.class })
   @SpecAssertion(section = "7.6")
   public void testFiresAnnotationOnInjectionPointWithTypeVariabledTypeParameterFails()
   {
      deployBeans(JavaSparrow_Broken.class);
      try
      {
         activateDependentContext();
         JavaSparrow_Broken bean = manager.getInstanceByType(JavaSparrow_Broken.class);
         assert bean != null;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6")
   public void testImplicitEventBeanMatchesAPITypeOfInectionPoint()
   {
      deployBeans(BlueFacedParrotFinch.class);
      try
      {
         activateDependentContext();
         BlueFacedParrotFinch bean = manager.getInstanceByType(BlueFacedParrotFinch.class);
         assert bean != null;
         // Retrieve the implicit event bean from the manager only by its API
         // type
         Set<?> eventBeans = manager.resolveByType(Event.class, new FiresBinding());
         assert !eventBeans.isEmpty();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6")
   public void testImplicitEventBeanMatchesBindingAnnotationsOfInjectionPoint()
   {
      deployBeans(OrangeCheekedWaxbill.class);
      try
      {
         activateDependentContext();
         OrangeCheekedWaxbill bean = manager.getInstanceByType(OrangeCheekedWaxbill.class);
         assert bean != null;
         // Retrieve the implicit event bean from the manager by its binding
         // types
         Set<?> eventBeans = manager.resolveByType(Event.class, new FiresBinding(), new TameAnnotationLiteral());
         assert !eventBeans.isEmpty();
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6")
   public void testImplicitEventBeanHasStandardDeploymentType()
   {
      deployBeans(BlueFacedParrotFinch.class);
      try
      {
         activateDependentContext();
         BlueFacedParrotFinch bean = manager.getInstanceByType(BlueFacedParrotFinch.class);
         assert bean != null;
         // Retrieve the implicit event bean from the manager only by its API
         // type
         Set<?> eventBeans = manager.resolveByType(Event.class, new FiresBinding());
         assert eventBeans.size() == 1;
         Bean<?> eventBean = (Bean<?>) eventBeans.iterator().next();
         assert eventBean.getDeploymentType().equals(Standard.class);
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6")
   public void testImplicitEventBeanHasDependentScope()
   {
      deployBeans(BlueFacedParrotFinch.class);
      try
      {
         activateDependentContext();
         BlueFacedParrotFinch bean = manager.getInstanceByType(BlueFacedParrotFinch.class);
         assert bean != null;
         // Retrieve the implicit event bean from the manager only by its API
         // type
         Set<?> eventBeans = manager.resolveByType(Event.class, new FiresBinding());
         assert eventBeans.size() == 1;
         Bean<?> eventBean = (Bean<?>) eventBeans.iterator().next();
         assert eventBean.getScopeType().equals(Dependent.class);
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testFireMethodCallsManagerFireWithEventObject()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testFireMethodCallsManagerFireWithBindingAnnotationsExceptObservable()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testFireMethodCallsManagerFireWithAllBindingAnnotationInstances()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testObserveMethodCallsManagerAddObserverWithObserverObject()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testObserveMethodCallsManagerAddObserverWithAllBindingAnnotationsExceptObservable()
   {
      assert false;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.6")
   public void testObserveMethodCallsManagerAddObserverWithAllBindingAnnotationInstance()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.7")
   public void testEventObjectContainsTypeVariablesWhenResolvingFails()
   {
      eventObjectContainsTypeVariables(new ArrayList<String>());
   }

   private <E> void eventObjectContainsTypeVariables(ArrayList<E> eventToFire)
   {
      @SuppressWarnings("unused")
      Set<?> resolvedObservers = manager.resolveObservers(eventToFire);
   }

   @Test(groups = { "broken", "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.7")
   public void testEventObjectContainsWildcardsWhenResolvingFails()
   {
      eventObjectContainsWildcards(new ArrayList<String>());
   }

   private void eventObjectContainsWildcards(ArrayList<? extends Object> eventToFire)
   {
      @SuppressWarnings("unused")
      //TODO There does not seem to be a way to get wildcarded types pass through
      Set<?> resolvedObservers = manager.resolveObservers(eventToFire);
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.7")
   public void testDuplicateBindingTypesWhenResolvingFails()
   {
      AnObserver observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class, new BindingTypeABinding());
      @SuppressWarnings("unused")
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType(), new BindingTypeABinding(), new BindingTypeABinding());
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.7")
   public void testNonBindingTypeAnnotationWhenResolvingFails()
   {
      AnObserver observer = new AnObserver();
      manager.addObserver(observer, new TypeLiteral<AnEventType>()
      {
      });
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType(), new AnimalStereotypeAnnotationLiteral());
      assert !resolvedObservers.isEmpty();
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.7")
   public void testResolvingChecksEventType()
   {
      Observer<AnEventType> observer = new AnObserver();
      manager.addObserver(observer, AnEventType.class);
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType());
      assert !resolvedObservers.isEmpty();
      Set<?> emptyObserverSet = manager.resolveObservers("A string event");
      assert emptyObserverSet.isEmpty();
   }

   @Test(groups = { "broken", "events" })
   @SpecAssertion(section = "7.7")
   public void testResolvingChecksTypeParameters()
   {
      AListObserver observer = new AListObserver();
      AnotherListObserver anotherObserver = new AnotherListObserver();
      manager.addObserver(observer, new TypeLiteral<ArrayList<String>>()
      {
      });
      manager.addObserver(anotherObserver, new TypeLiteral<ArrayList<Integer>>()
      {
      });
      Set<Observer<ArrayList<String>>> resolvedObservers = manager.resolveObservers(new ArrayList<String>());
      assert resolvedObservers.size() == 1;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.7")
   public void testResolvingChecksBindingTypes()
   {
      AnObserver observer = new AnObserver();
      AnObserver anotherObserver = new AnObserver();
      manager.addObserver(observer, AnEventType.class, new BindingTypeABinding());
      manager.addObserver(anotherObserver, AnEventType.class, new BindingTypeBBinding());
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType(), new BindingTypeABinding());
      assert resolvedObservers.size() == 1;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.7")
   public void testResolvingChecksBindingTypeMembers()
   {
      AnObserver observer = new AnObserver();
      AnObserver anotherObserver = new AnObserver();
      manager.addObserver(observer, AnEventType.class, new BindingTypeCBinding("first-observer"));
      manager.addObserver(anotherObserver, AnEventType.class, new BindingTypeCBinding("second-observer"));
      Set<Observer<AnEventType>> resolvedObservers = manager.resolveObservers(new AnEventType(), new BindingTypeCBinding("first-observer"));
      assert resolvedObservers.size() == 1;
   }

   @Test(groups = { "events", "inheritance" })
   @SpecAssertion(section = "4.")
   public void testNonStaticObserverMethodNotInherited()
   {
      deployBeans(LazyFarmer.class);
      assert manager.resolveObservers(new Egg()).isEmpty();
   }

   @Test
   @SpecAssertion(section = "7.5.8")
   public void testObserverCalledOnMostSpecializedInstance()
   {
      Shop.deliveryObservedBy = null;
      deployBeans(FarmShop.class, Shop.class);
      manager.fireEvent(new Delivery());
      assert Shop.deliveryObservedBy.equals(FarmShop.class.getName());
   }
}
