package org.jboss.webbeans.tck.unit.lookup.byname;

import java.lang.annotation.Annotation;
import java.util.List;

import javax.inject.AmbiguousDependencyException;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.testng.annotations.Test;

/**
 * 
 * Spec version: PRD2
 *
 */
public class ResolutionByNameTest extends AbstractTest
{
   
   @Override
   protected List<Class<? extends Annotation>> getEnabledDeploymentTypes()
   {
      List<Class<? extends Annotation>> deploymentTypes = super.getStandardDeploymentTypes();
      deploymentTypes.add(AnotherDeploymentType.class);
      return deploymentTypes;
   }

   /**
    * The getInstanceByName() method must:
    * 
    *   Identify the Web Bean by calling Manager.resolveByName(), passing the name.
    *   If resolveByName() returned an empty set, return a null value.
    *   Otherwise, if exactly one Web Bean was returned, obtain an instance of the 
    *    Web Bean by calling Manager.getInstance(), passing the Bean instance representing 
    *    the Web Bean
    */
   @Test(groups={"stub", "el"}) 
   @SpecAssertion(section="5.11")
   public void testELResolver()
   {           
      assert manager.getInstanceByName("nonexistant") == null;
                
      try
      {
         activateDependentContext();
         Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);      
         manager.addBean(salmonBean);      
         assert manager.getInstanceByName("salmon") instanceof Salmon;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   /**
    *   Otherwise, if resolveByName() returned more than one Web Bean, throw an 
    *    AmbiguousDependencyException.
    */
   @Test(expectedExceptions=AmbiguousDependencyException.class) 
   @SpecAssertion(section="5.11")
   public void testDuplicateNamedBeans()
   {
      try
      {
         activateDependentContext();
         
         Bean<Cod> codBean = createSimpleBean(Cod.class);
         Bean<Sole> soleBean = createSimpleBean(Sole.class);
         manager.addBean(codBean);
         manager.addBean(soleBean);
         
         assert manager.resolveByName("whitefish").size() == 2;
         manager.getInstanceByName("whitefish");
      }
      finally
      {
         deactivateDependentContext();
      }
   }
   
   @Test @SpecAssertion(section="5.11")
   public void testNamedBasedResolution()
   {
      
      Bean<Tuna> tunaBean = createSimpleBean(Tuna.class);
      Bean<Cod> codBean = createSimpleBean(Cod.class);
      Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);
      Bean<Sole> soleBean = createSimpleBean(Sole.class);
      Bean<SeaBass> seaBassBean = createSimpleBean(SeaBass.class);
      Bean<Haddock> haddockBean = createSimpleBean(Haddock.class);
      Bean<Plaice> plaiceBean = createSimpleBean(Plaice.class);
      
      manager.addBean(tunaBean);
      manager.addBean(codBean);
      manager.addBean(salmonBean);
      manager.addBean(soleBean);
      manager.addBean(haddockBean);
      manager.addBean(seaBassBean);
      
      assert manager.resolveByName("salmon").size() == 1;
      assert manager.resolveByName("salmon").contains(salmonBean);
      
      assert manager.resolveByName("whitefish").size() == 2;
      assert manager.resolveByName("whitefish").contains(codBean);
      assert manager.resolveByName("whitefish").contains(soleBean);
      
      manager.addBean(plaiceBean);
      
      assert manager.resolveByName("whitefish").size() == 1;
      assert manager.resolveByName("whitefish").contains(plaiceBean);
   }
   
   @Test(groups="resolution") @SpecAssertion(section="5.11")
   public void testNoWebBeansFound() throws Exception
   {
      Bean<Cod> codBean = createSimpleBean(Cod.class);
      Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);
      
      manager.addBean(codBean);
      manager.addBean(salmonBean);
      
      assert manager.resolveByName("foo").size() == 0;
   }
   
   /**
    * The following algorithm must be used by the Web Bean manager when resolving a Web Bean by name:
    * 
    *   The Web Bean manager identifies the set of matching enabled Web Beans which have the given name.
    *   Next, the Web Bean manager examines the deployment types of the matching Web Beans, as defined 
    *    in Section 2.5.7, Deployment type precedence, and returns the set of Web Beans with the 
    *    highest precedence deployment type that occurs in the set. If there are no matching Web Beans, 
    *    an empty set is returned.
    */
   @Test(groups="stub") @SpecAssertion(section="5.11")
   public void testOnlyHighestPrecedenceResolved()
   {
      // Both Cod and Plaice are named "whitefish" - Plaice has the highest deployment type (AnotherDeploymentType)
      Bean<Cod> codBean = createSimpleBean(Cod.class);
      Bean<Plaice> plaiceBean = createSimpleBean(Plaice.class);
      manager.addBean(codBean);
      manager.addBean(plaiceBean); 
      
      assert manager.resolveByName("whitefish").size() == 1;
      assert !manager.resolveByName("whitefish").contains(codBean);
      assert manager.resolveByName("whitefish").contains(plaiceBean);
   }   
}
