/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.webbeans.tck.unit.lookup.injectionpoint;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.context.Dependent;
import javax.inject.Current;
import javax.inject.Standard;
import javax.inject.manager.Bean;
import javax.inject.manager.InjectionPoint;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.jboss.webbeans.tck.impl.literals.CurrentBinding;
import org.testng.annotations.Test;

/**
 * Injection point metadata tests
 * 
 * @author David Allen
 * 
 * Spec version: PRD2
 */
public class InjectionPointTest extends AbstractTest
{

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetBean()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;

         Set<Bean<FieldInjectionPointBean>> resolvedBeans = manager.resolveByType(FieldInjectionPointBean.class);
         assert resolvedBeans.size() == 1;
         assert beanWithInjectionPoint.getInjectedMetadata().getBean().equals(resolvedBeans.iterator().next());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetType()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetBindingTypes()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         Set<Annotation> bindingTypes = beanWithInjectionPoint.getInjectedMetadata().getBindings();
         assert bindingTypes.size() == 1;
         assert Current.class.isAssignableFrom(bindingTypes.iterator().next().annotationType());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetMemberField()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert Field.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetMemberMethod()
   {
      deployBeans(MethodInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         MethodInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(MethodInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert Method.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());

         // Since the type and bindings must correspond to the parameter, check
         // them
         assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
         assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetMemberConstructor()
   {
      deployBeans(ConstructorInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         ConstructorInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(ConstructorInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert Constructor.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());

         // Since the type and bindings must correspond to the parameter, check
         // them
         assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
         assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetAnnotation()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert beanWithInjectionPoint.getInjectedMetadata().getAnnotation(AnimalStereotype.class) != null;
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testGetAnnotations()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         Set<Annotation> annotations = new HashSet<Annotation>(Arrays.asList(beanWithInjectionPoint.getInjectedMetadata().getAnnotations()));
         assert annotations.size() > 0;
         assert annotations.contains(new CurrentBinding());
         assert annotations.contains(new AnimalStereotypeAnnotationLiteral());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testStandardDeployment() throws Exception
   {
      deployBeans();

      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            assert manager.resolveByType(InjectionPoint.class).size() == 1;
            assert manager.resolveByType(InjectionPoint.class).iterator().next().getDeploymentType().equals(Standard.class);
         }
         
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testDependentScope() throws Exception
   {
      deployBeans();

      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            assert manager.resolveByType(InjectionPoint.class).size() == 1;
            assert manager.resolveByType(InjectionPoint.class).iterator().next().getScopeType().equals(Dependent.class);
         }
         
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testApiTypeInjectionPoint()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert InjectionPoint.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getClass());
      }
      finally
      {
         deactivateDependentContext();
      }
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6")
   public void testCurrentBinding()
   {
      deployBeans(FieldInjectionPointBean.class, BeanWithInjectionPointMetadata.class);

      // Get an instance of the bean which has another bean injected into it
      try
      {
         activateDependentContext();
         FieldInjectionPointBean beanWithInjectedBean = manager.getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
         BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
         assert beanWithInjectionPoint.getInjectedMetadata() != null;
         assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
      }
      finally
      {
         deactivateDependentContext();
      }
   }
}
