/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.jwt.auth.principal;

import io.smallrye.jwt.auth.principal.JWTAuthContextInfo;
import io.smallrye.jwt.auth.principal.KeyLocationResolver;
import io.smallrye.jwt.auth.principal.ParseException;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.eclipse.microprofile.jwt.Claims;
import org.jboss.logging.Logger;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.NumericDate;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.JwtContext;
import org.jose4j.keys.resolvers.VerificationKeyResolver;
import org.jose4j.lang.UnresolvableKeyException;

public class DefaultJWTTokenParser {
    private static final Logger LOGGER = Logger.getLogger(DefaultJWTTokenParser.class);
    private static final String ROLE_MAPPINGS = "roleMappings";
    private static final Pattern CLAIM_PATH_PATTERN = Pattern.compile("\\/(?=(?:(?:[^\"]*\"){2})*[^\"]*$)");
    private volatile VerificationKeyResolver keyResolver;

    public JwtContext parse(String token, JWTAuthContextInfo authContextInfo) throws ParseException {
        try {
            JwtConsumerBuilder builder = new JwtConsumerBuilder().setRequireExpirationTime();
            if (authContextInfo.getMaxTimeToLiveSecs() != null) {
                builder.setRequireIssuedAt();
            }
            builder.setJwsAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.WHITELIST, new String[]{authContextInfo.getSignatureAlgorithm().getAlgorithm()}));
            if (authContextInfo.isRequireIssuer()) {
                builder.setExpectedIssuer(true, authContextInfo.getIssuedBy());
            } else {
                builder.setExpectedIssuer(false, null);
            }
            if (authContextInfo.getSignerKey() != null) {
                builder.setVerificationKey((Key)authContextInfo.getSignerKey());
            } else {
                builder.setVerificationKeyResolver(this.getKeyResolver(authContextInfo));
            }
            if (authContextInfo.getExpGracePeriodSecs() > 0) {
                builder.setAllowedClockSkewInSeconds(authContextInfo.getExpGracePeriodSecs());
            } else {
                builder.setEvaluationTime(NumericDate.fromSeconds((long)0L));
            }
            this.setExpectedAudience(builder, authContextInfo);
            JwtConsumer jwtConsumer = builder.build();
            JwtContext jwtContext = jwtConsumer.process(token);
            JwtClaims claimsSet = jwtContext.getJwtClaims();
            this.verifyTimeToLive(authContextInfo, claimsSet);
            claimsSet.setClaim(Claims.raw_token.name(), (Object)token);
            if (!claimsSet.hasClaim(Claims.sub.name())) {
                String sub = this.findSubject(authContextInfo, claimsSet);
                claimsSet.setClaim(Claims.sub.name(), (Object)sub);
            }
            if (authContextInfo.isRequireNamedPrincipal()) {
                this.checkNameClaims(jwtContext);
            }
            if (!claimsSet.hasClaim(Claims.groups.name())) {
                List<String> groups = this.findGroups(authContextInfo, claimsSet);
                claimsSet.setClaim(Claims.groups.name(), groups);
            }
            if (claimsSet.hasClaim(ROLE_MAPPINGS)) {
                this.mapRoles(claimsSet);
            }
            return jwtContext;
        }
        catch (InvalidJwtException e) {
            LOGGER.debug((Object)"Token is invalid");
            throw new ParseException("Failed to verify a token", e);
        }
        catch (UnresolvableKeyException e) {
            LOGGER.debug((Object)"Verification key is unresolvable");
            throw new ParseException("Failed to verify a token", e);
        }
    }

    void setExpectedAudience(JwtConsumerBuilder builder, JWTAuthContextInfo authContextInfo) {
        Set<String> expectedAudience = authContextInfo.getExpectedAudience();
        if (expectedAudience != null) {
            builder.setExpectedAudience(expectedAudience.toArray(new String[0]));
        } else {
            builder.setSkipDefaultAudienceValidation();
        }
    }

    private void checkNameClaims(JwtContext jwtContext) throws InvalidJwtException {
        boolean hasPrincipalClaim;
        JwtClaims claimsSet = jwtContext.getJwtClaims();
        boolean bl = hasPrincipalClaim = claimsSet.getClaimValue(Claims.sub.name()) != null || claimsSet.getClaimValue(Claims.upn.name()) != null || claimsSet.getClaimValue(Claims.preferred_username.name()) != null;
        if (!hasPrincipalClaim) {
            throw new InvalidJwtException("No claim exists in sub, upn or preferred_username", Collections.emptyList(), jwtContext);
        }
    }

    private String findSubject(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) {
        if (authContextInfo.getSubjectPath() != null) {
            String[] pathSegments = DefaultJWTTokenParser.splitClaimPath(authContextInfo.getSubjectPath());
            Object claimValue = this.findClaimValue(authContextInfo.getSubjectPath(), claimsSet.getClaimsMap(), pathSegments, 0);
            if (claimValue instanceof String) {
                return (String)claimValue;
            }
            LOGGER.debugf("Claim value at the path %s is not a String", (Object)authContextInfo.getSubjectPath());
        }
        if (authContextInfo.getDefaultSubjectClaim() != null) {
            return authContextInfo.getDefaultSubjectClaim();
        }
        return null;
    }

    private List<String> findGroups(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) {
        if (authContextInfo.getGroupsPath() != null) {
            String[] pathSegments = DefaultJWTTokenParser.splitClaimPath(authContextInfo.getGroupsPath());
            Object claimValue = this.findClaimValue(authContextInfo.getGroupsPath(), claimsSet.getClaimsMap(), pathSegments, 0);
            if (claimValue instanceof List) {
                List groups = (List)List.class.cast(claimValue);
                try {
                    return Arrays.asList(groups.toArray(new String[0]));
                }
                catch (ArrayStoreException ex) {
                    LOGGER.debugf("Claim value at the path %s is not an array of strings", (Object)authContextInfo.getGroupsPath());
                }
            } else {
                if (claimValue instanceof String) {
                    return Arrays.asList(((String)claimValue).split(authContextInfo.getGroupsSeparator()));
                }
                LOGGER.debugf("Claim value at the path %s is neither an array of strings nor string", (Object)authContextInfo.getGroupsPath());
            }
        }
        if (authContextInfo.getDefaultGroupsClaim() != null) {
            return Collections.singletonList(authContextInfo.getDefaultGroupsClaim());
        }
        return null;
    }

    private static String[] splitClaimPath(String claimPath) {
        String[] stringArray;
        if (claimPath.indexOf(47) > 0) {
            stringArray = CLAIM_PATH_PATTERN.split(claimPath);
        } else {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = claimPath;
        }
        return stringArray;
    }

    private void mapRoles(JwtClaims claimsSet) {
        try {
            Map rolesMapping = (Map)claimsSet.getClaimValue(ROLE_MAPPINGS, Map.class);
            List groups = claimsSet.getStringListClaimValue(Claims.groups.name());
            ArrayList allGroups = new ArrayList(groups);
            for (Map.Entry mapping : rolesMapping.entrySet()) {
                if (!groups.contains(mapping.getKey())) continue;
                allGroups.add(mapping.getValue());
            }
            claimsSet.setStringListClaim(Claims.groups.name(), allGroups);
            LOGGER.tracef("Updated groups to: %s", allGroups);
        }
        catch (Exception e) {
            LOGGER.debug((Object)"Failed to access rolesMapping claim", (Throwable)e);
        }
    }

    private Object findClaimValue(String claimPath, Map<String, Object> claimsMap, String[] pathArray, int step) {
        Object claimValue = claimsMap.get(pathArray[step].replace("\"", ""));
        if (claimValue == null) {
            LOGGER.debugf("No claim exists at the path %s at segment %s", (Object)claimPath, (Object)pathArray[step]);
        } else if (step + 1 < pathArray.length) {
            if (claimValue instanceof Map) {
                Map nextMap = (Map)claimValue;
                int nextStep = step + 1;
                return this.findClaimValue(claimPath, nextMap, pathArray, nextStep);
            }
            LOGGER.debugf("Claim value at the path %s is not a json object", (Object)claimPath);
            return null;
        }
        return claimValue;
    }

    private void verifyTimeToLive(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) throws ParseException {
        Long maxTimeToLiveSecs = authContextInfo.getMaxTimeToLiveSecs();
        if (maxTimeToLiveSecs != null) {
            NumericDate exp;
            NumericDate iat;
            try {
                iat = claimsSet.getIssuedAt();
                exp = claimsSet.getExpirationTime();
            }
            catch (Exception e) {
                throw new ParseException("Failed to verify max TTL", e);
            }
            if (exp.getValue() - iat.getValue() > maxTimeToLiveSecs) {
                String msg = "The Expiration Time (exp=" + exp + ") claim value cannot be more than " + maxTimeToLiveSecs + " minutes in the future relative to Issued At (iat=" + iat + ")";
                throw new ParseException(msg);
            }
        } else {
            LOGGER.debugf("No max TTL has been specified in configuration", new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected VerificationKeyResolver getKeyResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        if (this.keyResolver == null) {
            DefaultJWTTokenParser defaultJWTTokenParser = this;
            synchronized (defaultJWTTokenParser) {
                if (this.keyResolver == null) {
                    this.keyResolver = new KeyLocationResolver(authContextInfo);
                }
            }
        }
        return this.keyResolver;
    }
}

