/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.dao.internal;

import org.jbpm.command.*;
import org.jbpm.ejb.LocalCommandService;
import org.jbpm.ejb.LocalCommandServiceHome;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.taskmgmt.exe.TaskInstance;
import org.jboss.bpm.console.server.util.ServiceLocator;

import javax.ejb.CreateException;
import java.util.List;

/**
 * Adopts a business interface to the command facade.<br>
 * Depends on the CommandServiceBean (<code>java:ejb/CommandServiceBean</code>).
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class JBPM3CommandDelegate
{

   private LocalCommandService facade;
   private static final String DEFAULT_JNDI_NAME = "java:ejb/CommandServiceBean";

   /**
    * Lookup with particluar JNDI name.
    * @param jndiName
    */
   public JBPM3CommandDelegate(String jndiName)
   {
      ejbLookup(jndiName);
   }

   /**
    * Lookup using default JNDI name.
    *
    * @see #DEFAULT_JNDI_NAME
    */

   public JBPM3CommandDelegate()
   {
      ejbLookup(DEFAULT_JNDI_NAME);
   }

   private void ejbLookup(String jndiName)
   {
      try
      {
         LocalCommandServiceHome home = (LocalCommandServiceHome)
           ServiceLocator.getEjbLocalHome(jndiName);
         this.facade = home.create();
      }
      catch (CreateException e)
      {
         throw new RuntimeException("Failed to create LocalCommandService", e);
      }
   }

   public List<ProcessDefinition> getActualDefinitions()
   {
      List<org.jbpm.graph.def.ProcessDefinition> defs = (List<org.jbpm.graph.def.ProcessDefinition>)
        facade.execute( new GetProcessDefinitionsCommand(false) );
      return defs;
   }

   public org.jbpm.graph.def.ProcessDefinition getActualDefinition(long processId)
   {
      List<org.jbpm.graph.def.ProcessDefinition> defs = getActualDefinitions();

      org.jbpm.graph.def.ProcessDefinition match = null;
      for(org.jbpm.graph.def.ProcessDefinition p0 : defs)
      {
         if(processId == p0.getId())
         {
            match = p0;
            break;
         }
      }

      if(null==match)
         throw new IllegalArgumentException("No process definition with ID " + processId);

      return match;
   }

   public void removeActualDefinition(long processId)
   {
      org.jbpm.graph.def.ProcessDefinition def = getActualDefinition(processId);
      facade.execute( new DeleteProcessdefinitionCommand(processId));
   }

   public org.jbpm.graph.exe.ProcessInstance getActualInstance(long instanceId)
   {
      org.jbpm.graph.exe.ProcessInstance instance = (org.jbpm.graph.exe.ProcessInstance)
        facade.execute( new GetProcessInstanceCommand(instanceId));
      return instance;
   }

   public List<org.jbpm.graph.exe.ProcessInstance> getActualInstances(long processId)
   {
      org.jbpm.graph.def.ProcessDefinition p0 = getActualDefinition(processId);
      GetProcessInstancesCommand command = new GetProcessInstancesCommand();
      command.setProcessId(processId);

      List<org.jbpm.graph.exe.ProcessInstance> instances =
        (List<org.jbpm.graph.exe.ProcessInstance>) facade.execute(command);

      return instances;
   }

   public org.jbpm.graph.exe.ProcessInstance startNewInstance(long processId)
   {
      org.jbpm.graph.def.ProcessDefinition p0 = getActualDefinition(processId);
      org.jbpm.graph.exe.ProcessInstance instance = (org.jbpm.graph.exe.ProcessInstance)
        facade.execute(new NewProcessInstanceCommand(p0.getName()));

      return instance;
   }

   public ProcessDefinition deploy(byte[] data)
   {
      org.jbpm.graph.def.ProcessDefinition p0 = (org.jbpm.graph.def.ProcessDefinition)
        facade.execute(
         new DeployProcessCommand(data)
      );
      return p0;  
   }

   public List<TaskInstance> getActualTasksForActors(List<String> actorList)
   {
      List<TaskInstance> tasks = (List<TaskInstance>)
            facade.execute( new GetTaskListCommand(actorList.toArray(new String[]{})));
      return tasks;
   }

   public TaskInstance getTaskById(long taskId)
   {

      // include variables, but no logs
      return (TaskInstance) facade.execute( new GetTaskInstanceCommand(taskId, true, false));
   }
}
