/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.ProcessDefinitionRefWrapper;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRefWrapper;
import org.jboss.bpm.console.server.gson.GsonFactory;
import org.jboss.bpm.console.server.integration.ExtensionManagement;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.ProcessManagement;

import com.google.gson.Gson;

/**
 * REST server module for accessing process related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("process")
public class ProcessMgmtFacade
{
   private static final Log log = LogFactory.getLog(ProcessMgmtFacade.class);
   private ProcessManagement processManagement;
   
   private ProcessManagement getProcessManagement()
   {
      if(null==this.processManagement)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.processManagement = factory.createProcessManagement();
         log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
      }

      return this.processManagement;
   }

   @GET
   @Path("definitions")
   @Produces("application/json")
   public Response getDefinitionsJSON()
   {
      ProcessDefinitionRefWrapper wrapper =
            new ProcessDefinitionRefWrapper(getProcessManagement().getProcessDefinitions());
      return createJsonResponse(wrapper);
   }

   @POST
   @Path("definitions/{id}/remove")
   @Produces("application/json")
   public Response removeDefinitionsJSON(
         @PathParam("id")
         long processId
   )
   {
      ProcessDefinitionRefWrapper wrapper =
            new ProcessDefinitionRefWrapper( getProcessManagement().removeProcessDefinition(processId));
      return createJsonResponse(wrapper);
   }

   @GET
   @Path("definitions/{id}/instances")
   @Produces("application/json")
   public Response getInstancesJSON(
         @PathParam("id")
         long processId
   )
   {

      ProcessInstanceRefWrapper wrapper =
            new ProcessInstanceRefWrapper(getProcessManagement().getProcessInstances(processId));
      return createJsonResponse(wrapper);
   }

   @POST
   @Path("definitions/{id}/instances/new")
   @Produces("application/json")
   public Response newInstance(
         @PathParam("id")
         long processId)
   {

      ProcessInstanceRef instance = getProcessManagement().newInstance(processId);
      return createJsonResponse(instance);
   }

   @POST
   @Path("instances/{id}/state/{next}")
   @Produces("application/json")
   public Response changeState(
         @PathParam("id")
         long instanceId,
         @PathParam("next")
         String next)
   {
      log.debug("Change instance (ID "+instanceId+") to state " +next);

      ProcessInstanceRef.STATE state = ProcessInstanceRef.STATE.valueOf(next);
      getProcessManagement().setProcessState(instanceId, state);
      return Response.ok().type("application/json").build();
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }
}
