/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import com.google.gson.Gson;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.TaskRefWrapper;
import org.jboss.bpm.console.client.model.forms.FieldDef;
import org.jboss.bpm.console.client.model.forms.FormDef;
import org.jboss.bpm.console.client.model.util.MapEntry;
import org.jboss.bpm.console.server.gson.GsonFactory;
import org.jboss.bpm.console.server.integration.ExtensionManagement;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.TaskManagement;
import org.jboss.bpm.console.server.integration.jbpm3.JBPM3FormParser;

import javax.ws.rs.*;
import javax.ws.rs.core.Response;
import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * REST server module for accessing task related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("tasks")
public class TaskMgmtFacade
{
   private static final Log log = LogFactory.getLog(TaskMgmtFacade.class);

   private TaskManagement taskManagement;
   private ExtensionManagement JBPM3Extension;

   /**
    * Lazy load the {@link org.jboss.bpm.console.server.integration.ProcessManagement}
    */
   private TaskManagement getTaskManagement()
   {
      if(null==this.taskManagement)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.taskManagement = factory.createTaskManagement();
         log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
      }

      return this.taskManagement;
   }

   private ExtensionManagement getManagementExtension()
   {
      if(null==this.JBPM3Extension)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.JBPM3Extension = factory.createExtensionManagement();
      }

      return this.JBPM3Extension;

   }

   @GET
   @Path("actor/{actorName}")
   @Produces("application/json")
   public Response getTasksForActor(
         @PathParam("actorName")
         String actorName
   )
   {
      TaskRefWrapper wrapper =
            new TaskRefWrapper(getTaskManagement().getTasksByActor(actorName));
      return createJsonResponse(wrapper);
   }

   @GET
   @Path("forms/{processId}")
   @Produces("application/json")
   public Response getTasksForms(
         @PathParam("processId")
         long processId
   )
   {
      Map<String,String> formMapping =
            getManagementExtension().getAvailableTaskForms(processId);

      List<MapEntry> entries = new ArrayList<MapEntry>();
      for(String s : formMapping.keySet())
      {
         entries.add( new MapEntry(s, formMapping.get(s)) );
      }

      return createJsonResponse( entries );
   }

   @POST
   @Path("{taskId}/assignment/{actorId}")
   @Produces("application/json")
   public Response assignTask(
         @PathParam("taskId")
         long taskId,
         @PathParam("actorId")
         String actorId
   )
   {
      log.debug("Reassign task " + taskId + " to '" + actorId +"'");
      getTaskManagement().reassignTask(taskId, actorId);
      return Response.ok().build();
   }

   @POST
   @Path("{taskId}/assignment")
   @Produces("application/json")
   public Response releaseTask(
         @PathParam("taskId")
         long taskId       
   )
   {
      log.debug("Release task " + taskId);
      getTaskManagement().reassignTask(taskId, null);
      return Response.ok().build();
   }

   @POST
   @Path("{taskId}/close/transition/default")
   @Produces("application/json")
   public Response closeTask(
         @PathParam("taskId")
         long taskId
   )
   {
      return closeTaskWithSignal(taskId, null);
   }

   @POST
   @Path("{taskId}/close/transition")
   @Produces("application/json")
   public Response closeTaskWithSignal(
         @PathParam("taskId")
         long taskId,
         @QueryParam("signal")
         String signalName
   )
   {
      log.debug("Close task " + taskId);
      getTaskManagement().closeTask(taskId, signalName);
      return Response.ok().build();
   }


   @GET
   @Path("forms/{processId}/{taskId}")
   @Produces("application/json")
   public Response getTasksFormByTaskId(
         @PathParam("processId")
         long processId,
         @PathParam("taskId")
         long taskId

   )
   {
      byte[] form = getManagementExtension().getTaskFormByTaskName(processId, taskId);
      JBPM3FormParser parser = new JBPM3FormParser();
      final FormDef formDef = new FormDef("taskform-"+taskId);
      try
      {
         parser.parse( new ByteArrayInputStream(form));

         parser.visitResults(
               new JBPM3FormParser.ParseResultVisitor()
               {

                  public void onInputField(FieldDef field)
                  {
                     formDef.getFields().add(field);
                  }

                  public void onInputButton(JBPM3FormParser.InputButton btn)
                  {
                     // TODO: parse buttons
                  }
               }
         );
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failed to parse form data", e);
      }

      return createJsonResponse(formDef);
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }
}
