/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.integration.jbpm3;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.forms.FieldDef;
import org.jboss.bpm.console.server.util.DOMUtils;
import org.w3c.dom.Element;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class JBPM3FormParser
{
   private static final Log log = LogFactory.getLog(JBPM3FormParser.class);

   private final String JSF_FACET_NAME = "facet";
   private final static String[] JSF_TEXT_NAMES = {"outputText", "inputText"};
   private final static String[] JSF_TEXTAREA_NAMES = { "outputTextarea","inputTextarea"};
   private final static String[] JSF_BOOLEAN_NAMES = {"selectBooleanCheckbox"};
   private final static String[] JSF_BUTTON_NAMES = {"saveButton", "transitionButton"};

   private List<FieldDef> inputFields = new ArrayList<FieldDef>();
   private List<InputButton> inputButtons = new ArrayList<InputButton>();

   static String test = "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n" +
         "\n" +
         "<html xmlns=\"http://www.w3.org/1999/xhtml\"\n" +
         "\txmlns:ui=\"http://java.sun.com/jsf/facelets\"\n" +
         "\txmlns:c=\"http://java.sun.com/jstl/core\"\n" +
         "\txmlns:h=\"http://java.sun.com/jsf/html\"\n" +
         "\txmlns:f=\"http://java.sun.com/jsf/core\"\n" +
         "\txmlns:tf=\"http://jbpm.org/jsf/tf\" xmlns:jbpm=\"http://jbpm.org/jsf\">\n" +
         "<ui:component>\n" +
         "\t<jbpm:dataform>\n" +
         "\t\t<f:facet name=\"header\">\n" +
         "\t\t\t<h:outputText value=\"#{taskName}\" />\n" +
         "\t\t</f:facet>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Purpose:\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:outputText value=\"#{var['purpose']}\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Description:\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:outputText value=\"#{var['purpose']}\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Self Paid\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:selectBooleanCheckbox value=\"#{var['selfPaid']}\" disabled=\"true\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Departure date\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:outputText value=\"#{var['departureDate']}\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Number of days\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:outputText value=\"#{var['daysNum']}\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Destination\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<h:outputText value=\"#{var['country']}\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "      <f:facet name=\"header\">\n" +
         "        <h:outputText value=\"City:\" />\n" +
         "      </f:facet>\n" +
         "      <h:outputText value=\"#{var['city']}\" />\n" +
         "    </jbpm:datacell>\n" +
         "\t\t<jbpm:datacell>\n" +
         "\t\t\t<f:facet name=\"header\">\n" +
         "\t\t\t\t<h:outputText value=\"Actions:\" />\n" +
         "\t\t\t</f:facet>\n" +
         "\t\t\t<tf:saveButton value=\"Save\" />\n" +
         "\t\t\t<tf:transitionButton value=\"Approve\" transition=\"approve\" />\n" +
         "\t\t\t<tf:transitionButton value=\"Reject\" transition=\"reject\" />\n" +
         "\t\t</jbpm:datacell>\n" +
         "\t</jbpm:dataform>\n" +
         "</ui:component>\n" +
         "</html>";

   public static void main(String[] args) throws Exception
   {
      JBPM3FormParser p = new JBPM3FormParser();
      p.parse( new ByteArrayInputStream(test.getBytes()));
      p.visitResults(
            new ParseResultVisitor()
            {
               public void onInputField(FieldDef field)
               {
                  log.debug(field);

               }

               public void onInputButton(InputButton btn)
               {
                  log.debug(btn);
               }
            }
      );
   }

   public void parse(InputStream in) throws Exception
   {
      Element root = DOMUtils.parse(in);
      Element comp = DOMUtils.getFirstChildElement(root, "component");
      Element dataForm = DOMUtils.getFirstChildElement(comp, "dataform");

      Iterator<Element> it = DOMUtils.getChildElements(dataForm, "datacell");
      while(it.hasNext())
      {
         parseDatacell(it.next());
      }

   }

   private String lastFieldName = null; // TODO: cleanup
   private void parseDatacell(Element datacell)
   {

      Iterator<Element> it = DOMUtils.getChildElements(datacell);
      while(it.hasNext())
      {
         Element next = it.next();

         // skip facet
         String nodeName = next.getLocalName();

         if(JSF_FACET_NAME.equals(nodeName))
         {
            lastFieldName = parseFieldName(next);
            continue;
         }

         for(String textName : JSF_TEXT_NAMES)
         {
            if(textName.equals(nodeName))
            {
               parseTextInputField(next);
               continue;
            }

         }

         for(String textName : JSF_TEXTAREA_NAMES)
         {
            if(textName.equals(nodeName))
            {
               parseTextAreaField(next);
               continue;
            }

         }

         for(String textName : JSF_BUTTON_NAMES)
         {
            if(textName.equals(nodeName))
            {
               parseButtonInputField(next);
               continue;
            }
         }

         for(String textName : JSF_BOOLEAN_NAMES)
         {
            if(textName.equals(nodeName))
            {
               parseBooleanInputField(next);
               continue;
            }
         }
      }

   }

   public void visitResults(ParseResultVisitor visitor)
   {
      for(FieldDef field : inputFields)
         visitor.onInputField(field);

      for(InputButton btn : inputButtons)
         visitor.onInputButton(btn);
   }

   public void reset()
   {
      this.inputButtons.clear();
      this.inputFields.clear();
   }


   private void parseTextAreaField(Element next)
   {
      String varName = getTrimmedVarname(next);
      inputFields.add( new FieldDef(lastFieldName,varName, FieldDef.InputType.TEXTAREA, String.class.getName(), true));
   }

   private String getTrimmedVarname(Element next)
   {
      String s = DOMUtils.getAttributeValue(next, "value");
      return trimVarName(s);
   }

   private void parseBooleanInputField(Element next)
   {
      String varName = getTrimmedVarname(next);
      inputFields.add( new FieldDef(lastFieldName,varName, FieldDef.InputType.BOOL, Boolean.class.getName(), false));
   }

   private void parseButtonInputField(Element next)
   {
      String name = getTrimmedVarname(next);
      String transition = DOMUtils.getAttributeValue(next, "transition");
      inputButtons.add( new InputButton(name, transition));
   }

   private void parseTextInputField(Element next)
   {
      String varName = getTrimmedVarname(next);
      inputFields.add( new FieldDef(lastFieldName, varName, FieldDef.InputType.TEXT, String.class.getName(), true));
   }

   private String parseFieldName(Element facet)
   {
      String fieldName = "UNKOWN_FIELDNAME";
      if("header".equals( facet.getAttribute("name")))
      {
         Element outputText = DOMUtils.getFirstChildElement(facet, "outputText");
         if(outputText!=null)
         {
            fieldName = DOMUtils.getAttributeValue(outputText, "value");
         }
      }

      return fieldName;
   }


   public final class InputButton
   {
      String name;
      String transition;

      public InputButton(String name, String transition)
      {
         this.name = name;
         this.transition = transition;
      }

      public boolean hasTransition()
      {
         return transition!=null;
      }

      public String toString()
      {
         return "Button{"+name+": "+transition+"}";
      }
   }

   public interface ParseResultVisitor
   {
      void onInputField(FieldDef field);
      void onInputButton(InputButton btn);
   }
   
   private static String trimVarName(String name)
   {
      String result = name;
      if(name.startsWith("#{var['"))
      {
         result = name.substring(7, name.length()-3);
      }

      return result;
   }
}
