/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client;

import com.google.gwt.core.client.GWT;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.Response;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTML;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.core.Margins;
import com.gwtext.client.core.RegionPosition;
import com.gwtext.client.widgets.*;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.form.FormPanel;
import com.gwtext.client.widgets.form.TextField;
import com.gwtext.client.widgets.layout.BorderLayout;
import com.gwtext.client.widgets.layout.BorderLayoutData;
import com.gwtext.client.widgets.layout.FitLayout;
import com.gwtext.client.widgets.layout.VerticalLayout;
import org.jboss.bpm.console.client.metric.MetricOverviewEditor;
import org.jboss.bpm.console.client.process.ProcessDefinitionListEditor;
import org.jboss.bpm.console.client.task.TaskListEditor;

import java.util.List;

/**
 * The main composite that assembles the gwt console application.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class MainView extends Composite
{
   private HTML status = new HTML();
   private Header header;
   private Menu menu;
   private Workspace workspace;
   private URLBuilder urlBuilder;

   public final static String[] KNOWN_ROLES = {"admin", "manager", "user"};

   private Viewport viewport;

   private Authentication auth;

   public MainView()
   {
      if(!GWT.isScript()) // hosted mode used proxy by default
         urlBuilder = new URLBuilder(GWT.getModuleBaseURL(), "xhp");
      else
         urlBuilder = new URLBuilder( "http://localhost:8080", "gwt-console-server"); // TODO: make configureable

      Panel mainPanel = createMainPanel();
      assembleMainApplication(mainPanel);
      forceLogin(mainPanel);
      viewport = new Viewport(mainPanel);
   }

   private Panel createMainPanel()
   {
      Panel mainPanel = new Panel();
      mainPanel.setBorder(false);
      mainPanel.setPaddings(5); // outer most padding
      mainPanel.setLayout(new FitLayout());
      mainPanel.setWidth(UIConstants.OVERALL_WIDTH);
      mainPanel.setHeight(UIConstants.OVERALL_WIDTH);
      mainPanel.hide();

      initWidget( mainPanel );
      return mainPanel;
   }

   private void assembleMainApplication(Panel mainPanel)
   {
      Panel borderPanel = new Panel();
      borderPanel.setBorder(false);
      borderPanel.setLayout(new BorderLayout());

      // ------------------------------------------

      header = new Header(this);
      borderPanel.add(header, new BorderLayoutData(RegionPosition.NORTH));

      // ------------------------------------------

      menu = new Menu(this);
      BorderLayoutData menuData = new BorderLayoutData(RegionPosition.WEST);
      menuData.setSplit(false);
      menuData.setMinSize(UIConstants.MAIN_MENU_MIN);
      menuData.setMaxSize(UIConstants.MAIN_MENU_MAX);
      menuData.setMargins(new Margins(0, 5, 0, 0));
      borderPanel.add(menu, menuData);

      // ------------------------------------------

      workspace = new Workspace();
      borderPanel.add(workspace, new BorderLayoutData(RegionPosition.CENTER));

      // ------------------------------------------

      workspace.addEditor( new ProcessDefinitionListEditor(this), false );
      //workspace.addEditor( new TaskListEditor(this), false );
      workspace.addEditor( new MetricOverviewEditor(this), false);

      // ------------------------------------------

      mainPanel.add(borderPanel);
   }

   private void forceLogin(final Panel mainPanel)
   {
      Panel panel = new Panel();
      panel.setPaddings(20);
      panel.setStyleName("login-panel-content");
      panel.setLayout(new VerticalLayout(10));
      panel.add( new HTML("Welcome, please login.") );

      final Window window = new Window();
      window.setTitle("GWT-Console");
      window.setClosable(false);
      window.setResizable(false);
      window.setWidth(300);
      window.setHeight(180);
      window.setLayout(new BorderLayout());
      window.setCloseAction(Window.CLOSE);

      final FormPanel loginForm = new FormPanel();
      loginForm.setBorder(false);
      loginForm.add(new TextField("Username", "user"));
      TextField passwordField = new TextField("Password", "pass");
      passwordField.setPassword(true);
      loginForm.add(passwordField);

      final Button submitBtn = new Button("Login",
            new ButtonListenerAdapter()
            {
               public void onClick(Button button, EventObject e)
               {
                  String user = loginForm.getForm().findField("user").getValueAsString();
                  String pass = loginForm.getForm().findField("pass").getValueAsString();

                  String url = getUrlBuilder().getUserInRoleURL(KNOWN_ROLES);
                  auth = new Authentication(url);
                  auth.setCallback(
                        new Authentication.AuthCallback()
                        {

                           public void onLoginSuccess(Request request, Response response)
                           {
                              // clear the form
                              loginForm.getForm().reset();
                              
                              // display main console
                              window.close();
                              mainPanel.show();
                              mainPanel.doLayout();

                              // update header
                              header.setUserInfo(auth.getUsername(), auth.getRolesAssigned());
                           }

                           public void onLoginFailed(Request request, Throwable t)
                           {
                              // auth failed
                              setError("Authentication failed." +t.getMessage());
                              t.printStackTrace(System.out);
                           }
                        }
                  );

                  auth.doLogin(user,pass);

               }
            });

      loginForm.addButton(submitBtn);
      panel.add(loginForm);
      
      BorderLayoutData centerData = new BorderLayoutData(RegionPosition.CENTER);
      centerData.setMargins(3, 0, 3, 3);

      window.add(panel, centerData);
      // ------------------------------------------

      window.show();
   }


   public void addEditor(Editor editor)
   {
      workspace.addEditor(editor,  true);
   }

   public boolean hasEditor(String id)
   {
      return workspace.hasEditor(id);
   }

   public void showEditor(String id)
   {
      workspace.showTab(id);
   }

   public URLBuilder getUrlBuilder()
   {
      return this.urlBuilder;
   }

   public void setError(final String error)
   {
      displayMessage(error, true);
   }

   public void displayMessage(final String message, final boolean isError)
   {
      status.setText(message);

      MessageBox.show(new MessageBoxConfig() {
         {
            String title = isError ? "Unknown error" : "System Message";
            setTitle(title);
            setMsg(message);
            setButtons(MessageBox.OK);
            setCallback(new MessageBox.PromptCallback()
            {
               public void execute(String btnID, String text)
               {
               }
            });
         }
      });
   }

   public List<String> getRolesAssigned()
   {
      return auth.getRolesAssigned();
   }

   public String getUsername()
   {
      return auth.getUsername();
   }

   public void logout()
   {
      auth.doLogout();
   }
}
