/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.http.client.*;
import com.gwtext.client.data.*;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.MessageBoxConfig;
import com.gwtext.client.widgets.Component;
import com.gwtext.client.widgets.event.PanelListenerAdapter;
import com.gwtext.client.widgets.grid.ColumnConfig;
import com.gwtext.client.widgets.grid.ColumnModel;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.model.ProcessDefinition;
import org.jboss.bpm.console.client.widgets.RemoteListView;

import java.util.*;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class ProcessDefinitionList extends RemoteListView
{
   private Map<Integer, ProcessDefinition> row2ProcessMap = new HashMap<Integer,ProcessDefinition>();

   public ProcessDefinitionList(String titleName, final MainView view)
   {
      super(titleName, view, getResourceUrl(view));
      enableAddBtn(false);
   }

   private static String getResourceUrl(MainView view)
   {
      return view.getUrlBuilder().getProcessDefinitionURL();
   }

   protected ColumnModel createColumnModel()
   {
      final ColumnModel columnModel = new ColumnModel(
        new ColumnConfig[]
          {
            new ColumnConfig("Process ID", "processId", 75, true),
            new ColumnConfig("Name", "name", 300, true, null, "expand"),
            new ColumnConfig("Version", "version", 75, true)
          }
      );
      return columnModel;
   }

   protected JsonReader createReader()
   {
      final RecordDef recordDef = new RecordDef(
        new FieldDef[]{
          new IntegerFieldDef("processId"),
          new StringFieldDef("name"),
          new StringFieldDef("version")
        }
      );

      final JsonReader reader = new JsonReader(recordDef);
      reader.setRoot("definitions");
      reader.setTotalProperty("totalCount");
      reader.setId("processId");
      return reader;
   }

   protected void onExamine(int row)
   {
      ProcessDefinition proc = row2ProcessMap.get(row);
      String editorId = ProcessInstanceEditor.createWidgetID(proc);

      if(view.hasEditor(editorId))
         view.showEditor(editorId);
      else
         view.addEditor( new ProcessInstanceListEditor(proc, view) );
   }

   protected void onDelete(final int row)
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("Delete process?");
            setMsg("Deleting the process will remove all instances as well. " +
              "Would you like to delete this process?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     ProcessDefinition proc = (ProcessDefinition)row2ProcessMap.get(row);
                     String url = view.getUrlBuilder().getRemoveDefinitionURL(proc.getProcessId());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                reloadStore();
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                view.setError("Failed to remove process definition: "+t.getMessage());
                                t.printStackTrace(System.out);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        view.setError(e1.getMessage());
                        e1.printStackTrace();
                     }
                  }
               }
            });
         }
      });
   }


   protected void onAdd()
   {
      
   }

   protected void onRecordsLoaded(Record[] records)
   {
      int i = 0;
      for(Record r : records)
      {
         Long id = Long.valueOf(r.getAsString("processId"));
         ProcessDefinition pd = new ProcessDefinition(
           id,
           r.getAsString("name"),
           r.getAsString("version")
         );

         row2ProcessMap.put(i, pd);
         i++;
      }

      System.out.println("Loaded " + row2ProcessMap.size() + " process definitions");
   }

   public List<ProcessDefinition> getAvailableProcessDefinitions()
   {
      List<ProcessDefinition> results = new ArrayList<ProcessDefinition>();
      Iterator<Integer> keys = row2ProcessMap.keySet().iterator();
      while(keys.hasNext())
      {
         results.add(row2ProcessMap.get(keys.next()));
      }

      return results;
   }
}

