/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.http.client.*;
import com.gwtext.client.data.*;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.MessageBoxConfig;
import com.gwtext.client.widgets.grid.ColumnConfig;
import com.gwtext.client.widgets.grid.ColumnModel;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.util.DateRenderer;
import org.jboss.bpm.console.client.model.ProcessDefinition;
import org.jboss.bpm.console.client.model.ProcessInstance;
import org.jboss.bpm.console.client.widgets.RemoteListView;

import java.util.HashMap;
import java.util.Map;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class ProcessInstanceList extends RemoteListView
{

   private Map<Integer, ProcessInstance> row2InstanceMap = new HashMap<Integer, ProcessInstance>();
   private ProcessDefinition parent;
   private static final String DATE_FORMAT = "y-m-j H:i:s";  //08-10-02 13:51:27

   public ProcessInstanceList(ProcessDefinition procDef, String titleName, MainView view)
   {
      super(titleName, view, getResourceUrl(view, procDef.getProcessId()));
      this.parent = procDef;
   }

   ProcessDefinition getProcessDefinition()
   {
      return parent;
   }

   protected void onExamine(final int row)
   {
      ProcessInstance instance = row2InstanceMap.get(row);
      String id = ProcessInstanceEditor.ID+"."+instance.getInstanceId();
      if(view.hasEditor(id))
         view.showEditor(id);
      else
         view.addEditor( new ProcessInstanceEditor(parent, instance, view) );
   }

   protected void onDelete(final int row)
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("Delete process instance?");
            setMsg("Would you like to delete this process instance?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     ProcessInstance instance = row2InstanceMap.get(row);
                     instance.setState(ProcessInstance.STATE.ENDED);
                     String url = view.getUrlBuilder().getStateChangeURL(instance.getInstanceId(), instance.getState());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                reloadStore();
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                view.setError("Failed to remove process instance: "+t.getMessage());
                                t.printStackTrace(System.out);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        view.setError(e1.getMessage());
                        e1.printStackTrace();
                     }
                  }
               }
            });
         }
      });
   }


   protected void onAdd()
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("New instance");
            setMsg("Would you like to start a new instance?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     String url = view.getUrlBuilder().getStartNewInstanceURL(parent.getProcessId());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                reloadStore();
                                doLayout();
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                view.setError("Failed to start new instance: "+t.getMessage());
                                t.printStackTrace(System.out);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        view.setError(e1.getMessage());
                        e1.printStackTrace();
                     }
                  }
               }
            });
         }
      });
   }

   protected void onRecordsLoaded(Record[] records)
   {
      int i = 0;
      for(Record r : records)
      {
         ProcessInstance pd = transform(r);
         row2InstanceMap.put(i, pd);
         i++;
      }

      System.out.println("Loaded " + row2InstanceMap.size() + " records");
   }

   public static ProcessInstance transform(Record r)
   {
      ProcessInstance pd = new ProcessInstance(
        Long.valueOf(r.getAsString("instanceId")),
        Long.valueOf( r.getAsString("parentId")),
        r.getAsDate("startDate"),
        r.getAsDate("endDate"),
        r.getAsBoolean("suspended"));
      return pd;
   }

   protected ColumnModel createColumnModel()
   {      
      final ColumnModel columnModel = new ColumnModel(
        new ColumnConfig[]
          {
            new ColumnConfig("Instance ID", "instanceId", 75, true),
            //new ColumnConfig("Process ID", "parentId", 25, true),
            //new ColumnConfig("Key", "key", 50, true),
            new ColumnConfig("State", "state", 100, true, null, "expand"),
            new ColumnConfig("Start Date", "startDate", 125, true, new DateRenderer("startDate")),
            new ColumnConfig("End Date", "endDate", 125, true, new DateRenderer("endDate"))
          }
      );

      return columnModel;
   }

   protected JsonReader createReader()
   {
      final RecordDef recordDef = new RecordDef(
        new FieldDef[]{
          new IntegerFieldDef("instanceId"),
          new IntegerFieldDef("parentId"),
          new StringFieldDef("key"),
          new StringFieldDef("state"),
          new DateFieldDef("startDate", DATE_FORMAT),
          new DateFieldDef("endDate", DATE_FORMAT),
          new BooleanFieldDef("suspended")
        }
      );

      final JsonReader reader = new JsonReader(recordDef);
      reader.setRoot("instances");
      reader.setTotalProperty("totalCount");
      reader.setId("instanceId");

      return reader;
   }

   private static String getResourceUrl(MainView view, long parentId)
   {
      return view.getUrlBuilder().getProcessInstanceURL(parentId);
   }

}
