/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.http.client.*;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.core.Function;
import com.gwtext.client.core.Position;
import com.gwtext.client.data.Record;
import com.gwtext.client.widgets.Button;
import com.gwtext.client.widgets.PaddedPanel;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.layout.ColumnLayout;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.form.DateField;
import com.gwtext.client.widgets.form.FieldSet;
import com.gwtext.client.widgets.form.FormPanel;
import com.gwtext.client.widgets.form.TextField;
import com.gwtext.client.widgets.grid.RowSelectionModel;
import com.gwtext.client.widgets.grid.event.RowSelectionListenerAdapter;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.Editor;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.model.ProcessDefinition;
import org.jboss.bpm.console.client.model.ProcessInstance;
import org.jboss.bpm.console.client.widgets.TeaserPanel;
import org.jboss.bpm.console.client.widgets.HelpPanel;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 *
 * @see org.jboss.bpm.console.client.process.ProcessDefinitionList#onExamine(int)
 */
public class ProcessInstanceListEditor extends Editor
{

   private ProcessDefinition parent;
   private ProcessInstance selectedInstance;
   private MainView view;
   private ProcessInstanceList instanceList;

   public ProcessInstanceListEditor(ProcessDefinition proc, MainView view)
   {
      super();

      this.parent = proc;
      this.view = view;
      this.setId( ProcessInstanceEditor.createWidgetID(proc) );

      // --------------------------------------------------

      this.setLayout( new ColumnLayout() );
      this.setWidth(UIConstants.EDITOR_WIDTH);

      // ---------------

      Panel leftHand = new Panel();
      leftHand.setFrame(false);
      leftHand.setHeader(false);
      leftHand.setBorder(false);
      
      instanceList = new ProcessInstanceList(
        proc, "Process Instances", view
      );

      // ---------------

      Panel outerFormPanel = new Panel();
      outerFormPanel.setBorder(false);
      outerFormPanel.setFrame(false);
      outerFormPanel.setPaddings(10);

      // ---------------

      final FormPanel formPanel = new FormPanel();
      formPanel.setLabelAlign(Position.LEFT);
      formPanel.setWidth(UIConstants.EDITOR_PANEL_WIDTH);
      formPanel.setHeader(false);
      formPanel.setFrame(false);
      formPanel.setPaddings(5, 5, 5, 0);
      formPanel.setLabelWidth(120);

      // ---------------

      FieldSet fieldSet = new FieldSet();
      fieldSet.setLabelWidth(90);
      fieldSet.setTitle("Process Instance Details");
      fieldSet.setAutoHeight(true);
      fieldSet.setBorder(false);

      // ---------------
      //the field names must match the data field values from the Store
      fieldSet.add(new TextField("Id", "instanceId", 230));
      fieldSet.add(new TextField("Key", "key", 230));
      fieldSet.add(new TextField("State", "state", 230));
      fieldSet.add(new DateField("Start Date", "startDate", 230));
      fieldSet.add(new DateField("End Date", "endDate", 230));
      Panel inner = new PaddedPanel(fieldSet, 0, 10, 0, 0);

      formPanel.add(inner);

      final Button suspendButton = new Button("Suspend",
        new ButtonListenerAdapter()
        {

           public void onClick(Button button, EventObject eventObject)
           {
              selectedInstance.setState(ProcessInstance.STATE.SUSPENDED);
              persistStateChange();
           }
        }       
      );                        
      //suspendButton.setIcon("images/icons/pause.png");

      final Button resumeButton = new Button("Resume",
        new ButtonListenerAdapter()
        {

           public void onClick(Button button, EventObject eventObject)
           {
              selectedInstance.setState(ProcessInstance.STATE.RUNNING);
              persistStateChange();
           }
        }
      );
      //resumeButton.setIcon("images/icons/play.png");
      
      formPanel.addButton(suspendButton);
      formPanel.addButton(resumeButton);

      // ----------------

      final RowSelectionModel sm = new RowSelectionModel(true);
      sm.addListener(
        new RowSelectionListenerAdapter()
        {
           public void onRowSelect(RowSelectionModel sm, int rowIndex, Record record)
           {
              selectedInstance = ProcessInstanceList.transform(record);
              if(selectedInstance.isSuspended())
              {
                 suspendButton.disable();
                 resumeButton.enable();
              }
              else
              {
                 resumeButton.disable();
                 suspendButton.enable();
              }

              formPanel.getForm().loadRecord(record);
              formPanel.doLayout();
           }
        }
      );
      instanceList.getGrid().setSelectionModel(sm);

      //select the first row after a little delay so that the rows are rendered
      instanceList.getGrid().doOnRender(
        new Function()
        {
           public void execute() {
              sm.selectFirstRow();
           }
        }, 10);


      outerFormPanel.add(formPanel);

      // ----------------------------------
      
      HelpPanel help = new HelpPanel(UIConstants.TEASER_PANEL_WIDTH, 200, "Managing process instances");      

      TeaserPanel teaserPanel = new TeaserPanel();
      teaserPanel.add(help);

      // ----------------------------------

      leftHand.add(instanceList);
      leftHand.add(outerFormPanel);
      
      this.add(leftHand, new ColumnLayoutData(0.7) );
      this.add(teaserPanel , new ColumnLayoutData(0.3) );      

   }

   private void persistStateChange()
   {
      RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, view.getUrlBuilder().getStateChangeURL(
        selectedInstance.getInstanceId(), selectedInstance.getState())
      );

      try
      {
         rb.sendRequest(null,
           new RequestCallback() {

              public void onResponseReceived(Request request, Response response)
              {
                 // testSuccess
                 instanceList.reloadStore();
                 selectedInstance = null;
              }

              public void onError(Request request, Throwable t)
              {
                 // Unknown error
                 view.setError(t.getMessage());
                 t.printStackTrace(System.out);
              }
           });
      }
      catch (RequestException e1)
      {
         view.setError(e1.getMessage());
         e1.printStackTrace();
      }
   }


   public String getEditorId()
   {
      return getId();
   }

   public String getTitle()
   {
      return parent.getName()+" (#"+parent.getVersion()+")";
   }


   public String getIconCSS()
   {
      return "bpm-instance-icon";
   }
}

