/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.widgets;

import com.gwtext.client.core.EventObject;
import com.gwtext.client.data.*;
import com.gwtext.client.data.event.StoreListenerAdapter;
import com.gwtext.client.widgets.*;
import com.gwtext.client.widgets.event.PanelListenerAdapter;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.grid.ColumnModel;
import com.gwtext.client.widgets.grid.GridPanel;
import com.gwtext.client.widgets.grid.event.GridCellListener;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;

/**
 * A {@link com.gwtext.client.widgets.grid.GridPanel} view that will pull data
 * from a remote REST/Json resource.
 * <p>
 * The view will be lazily initialized by calling {@link com.gwtext.client.data.Store#load()}
 * in {@link com.gwtext.client.widgets.event.PanelListenerAdapter#doBeforeRender(com.gwtext.client.widgets.Component)}
 * and subsequently add the grid to the container panel upon {@link com.gwtext.client.data.event.StoreListenerAdapter#onLoad(com.gwtext.client.data.Store, com.gwtext.client.data.Record[])}
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public abstract class RemoteListView extends Panel
{
   protected int selectedRowIndex = -1;
   protected MainView view;
   protected static final int PAGE_SIZE = 25;
   private GridPanel grid;
   protected String title;

   private Store store;
   private PagingToolbar pagingToolbar;

   private boolean storeInit = false;

   private ToolbarButton addButton;
   private ToolbarButton deleteButton;

   public RemoteListView(String titleName, MainView view, String resourceUrl)
   {
      super();

      this.title = titleName;
      this.view = view;

      // ----------------------------------------

      this.setPaddings(10);
      this.setHeader(false);
      this.setBorder(false);
      this.setFrame(false);

      // ----------------------------------------

      grid = createGridPanel(titleName);

      // ----------------------------------------

      final JsonReader reader = createReader();

      // ----------------------------------------

      final ColumnModel columnModel = createColumnModel();
      grid.setColumnModel(columnModel);
      grid.setAutoExpandColumn("expand");
      //grid.setAutoExpandMin(300);

      // ----------------------------------------

      DataProxy dataProxy = new ScriptTagProxy(resourceUrl, 1000*10);

      store = new Store(dataProxy, reader, false); // boolean==remoteSort, add's GET parameters      
      store.addStoreListener( new ListViewStoreListener(this, grid) );
      grid.setStore(store);

      // ------------------------------------------
      pagingToolbar = createToolbar(store);
      grid.setBottomToolbar(pagingToolbar);

      // PanelListener will lazy load store data.
      // Store.onLoad() callback will populate the grid
      // and add it to the container panel.      
      this.addListener( new PanelListenerAdapter()
      {
         public boolean doBeforeRender(Component component)
         {
            System.out.println("Reload store trigger");
            reloadStore();
            return true;
         }
      }
      );

   }

   protected void enableAddBtn(boolean isEnabled)
   {
      if(isEnabled)
         this.addButton.enable();
      else
         this.addButton.disable();
   }

   protected void enableDeleteBtn(boolean isEnabled)
   {
      if(isEnabled)
         this.deleteButton.enable();
      else
         this.deleteButton.disable();
   }

   private GridPanel createGridPanel(String titleName)
   {
      GridPanel grid = new GridPanel();

      grid.setTitle(titleName);
      grid.setFrame(true);
      grid.setWidth(UIConstants.EDITOR_PANEL_WIDTH);
      grid.stripeRows(true);
      grid.setLoadMask(true);

      grid.addGridCellListener( new ListViewCellListener() );

      return grid;
   }

   private PagingToolbar createToolbar(final Store store)
   {
      PagingToolbar pagingToolbar = new PagingToolbar(store);
      pagingToolbar.setPageSize(PAGE_SIZE);
      pagingToolbar.setDisplayInfo(true);
      pagingToolbar.setDisplayMsg("{0} - {1} of {2}");
      //pagingToolbar.setEmptyMsg("No entities to display");

      pagingToolbar.addSeparator();
      pagingToolbar.addSpacer();

      ToolbarButton examineButton = new ToolbarButton("Examine", new ButtonListenerAdapter()
      {
         public void onClick(Button button, EventObject e)
         {
            if (-1 == selectedRowIndex)
               MessageBox.alert("Please select a process.");
            else
               onExamine(selectedRowIndex);
         }
      });
      //examineButton.setIcon("images/icons/examine.png");
      examineButton.setTooltip("Examine");
      pagingToolbar.addButton(examineButton);


      addButton = new ToolbarButton("Add", new ButtonListenerAdapter()
      {
         public void onClick(Button button, EventObject e)
         {
            onAdd();
         }
      });
      //addButton.setIcon("images/icons/add.png");
      addButton.setTooltip("Add");
      pagingToolbar.addButton( addButton );

      deleteButton = new ToolbarButton("Remove", new ButtonListenerAdapter()
      {
         public void onClick(Button button, EventObject e)
         {
            if (-1 == selectedRowIndex)
            {
               MessageBox.alert("Please select a process.");
            } else
            {
               onDelete(selectedRowIndex);
            }
         }
      });
      //deleteButton.setIcon("images/icons/remove.png");
      deleteButton.setTooltip("Remove");
      pagingToolbar.addButton(deleteButton);

      return pagingToolbar;
   }

   protected abstract void onAdd();

   protected abstract void onExamine(final int row);

   protected abstract void onDelete(final int row);

   protected abstract void onRecordsLoaded(Record[] records);

   protected abstract ColumnModel createColumnModel();

   protected abstract JsonReader createReader();

   private class ListViewStoreListener extends StoreListenerAdapter
   {

      Panel containerPanel;
      GridPanel grid;

      public ListViewStoreListener(Panel containerPanel, GridPanel grid)
      {
         this.containerPanel = containerPanel;
         this.grid = grid;
      }

      /**
       * load callback will push records to implementation
       * and (!) add the grid to the container panel.
       */
      public void onLoad(Store store, Record[] records)
      {
         onRecordsLoaded(records);

         if(!storeInit)
         {
            containerPanel.clear();
            containerPanel.add(grid);
            storeInit = true;
         }
         containerPanel.doLayout();
      }

      public void onLoadException(Throwable throwable)
      {
         throwable.printStackTrace(System.out);
         view.setError("Failed to load remote data: " + throwable.getMessage());
      }

   };

   private class ListViewCellListener implements GridCellListener
   {
      public void onCellClick(GridPanel grid, int rowIndex, int colindex, EventObject e)
      {
         selectedRowIndex = rowIndex;
      }

      public void onCellContextMenu(GridPanel gridPanel, int i, int i1, EventObject eventObject)
      {

      }

      public void onCellDblClick(GridPanel gridPanel, int i, int i1, EventObject eventObject)
      {

      }
   }

   public GridPanel getGrid()
   {
      return grid;
   }

   public void reloadStore()
   {
      store.load(0,PAGE_SIZE);
   }
}
