/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client;

import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTML;
import com.gwtext.client.core.Margins;
import com.gwtext.client.core.RegionPosition;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.MessageBoxConfig;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.Viewport;
import com.gwtext.client.widgets.layout.BorderLayout;
import com.gwtext.client.widgets.layout.BorderLayoutData;
import com.gwtext.client.widgets.layout.FitLayout;
import org.jboss.bpm.console.client.process.ProcessDefinitionListEditor;
import org.jboss.bpm.console.client.report.OverviewReportEditor;
import org.jboss.bpm.console.client.task.TaskListEditor;
import org.jboss.bpm.console.client.util.ConsoleLog;

import java.util.List;

/**
 * The main composite that assembles the gwt console application.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class MainView extends Composite
{
   private HTML status = new HTML();
   private Header header;
   private Menu menu;
   private Workspace workspace;

   private URLBuilder urlBuilder;
   private Viewport viewport;
   private Authentication auth;

   private ConsoleConfig config;

   public boolean isAttached()
   {
      return super.isAttached();    //To change body of overridden methods use File | Settings | File Templates.
   }

   public MainView(Authentication auth, URLBuilder urlBuilder, ConsoleConfig config)
   {
      this.auth = auth;
      this.config = config;
      this.urlBuilder = urlBuilder;

      Panel mainPanel = createMainPanel();
      assembleMainApplication(mainPanel);

      viewport = new Viewport(mainPanel);
   }

   private Panel createMainPanel()
   {
      Panel mainPanel = new Panel();
      mainPanel.setBorder(false);
      mainPanel.setPaddings(5); // outer most padding
      mainPanel.setLayout(new FitLayout());
      mainPanel.setWidth(UIConstants.OVERALL_WIDTH);
      mainPanel.setHeight(UIConstants.OVERALL_WIDTH);      
      return mainPanel;
   }

   private void assembleMainApplication(Panel mainPanel)
   {
      Panel borderPanel = new Panel();
      borderPanel.setBorder(false);
      borderPanel.setLayout(new BorderLayout());

      // ------------------------------------------

      header = new Header(this);
      borderPanel.add(header, new BorderLayoutData(RegionPosition.NORTH));      
      header.setUserInfo(auth.getUsername(), auth.getRolesAssigned());
      
      // ------------------------------------------

      menu = new Menu(this);
      BorderLayoutData menuData = new BorderLayoutData(RegionPosition.WEST);
      menuData.setSplit(false);
      menuData.setMinSize(UIConstants.MAIN_MENU_MIN);
      menuData.setMaxSize(UIConstants.MAIN_MENU_MAX);
      menuData.setMargins(new Margins(0, 5, 0, 0));
      borderPanel.add(menu, menuData);

      // ------------------------------------------

      workspace = new Workspace();
      borderPanel.add(workspace, new BorderLayoutData(RegionPosition.CENTER));

      // ------------------------------------------
      // TODO: these don't initialze correctly when ordered differently or launched from Menu
      workspace.addEditor( new ProcessDefinitionListEditor(this), false );
      workspace.addEditor( new TaskListEditor(this), false );
      workspace.addEditor( new OverviewReportEditor(this), false);

      // ------------------------------------------

      mainPanel.add(borderPanel);
   }

   public void addEditor(Editor editor)
   {
      ConsoleLog.debug("Add editor " + editor.getEditorId());
      workspace.addEditor(editor,  true);
   }

   public boolean hasEditor(String id)
   {
      return workspace.hasEditor(id);
   }

   public void showEditor(String id)
   {      
      workspace.showTab(id);
   }

   public URLBuilder getUrlBuilder()
   {
      return this.urlBuilder;
   }

   public void displayMessage(final String message, final boolean isError)
   {
      status.setText(message);

      MessageBox.show(new MessageBoxConfig() {
         {
            String title = isError ? "Error" : "System Message";
            setTitle(title);
            setMsg(message);
            setButtons(MessageBox.OK);
            setCallback(new MessageBox.PromptCallback()
            {
               public void execute(String btnID, String text)
               {
               }
            });
         }
      });
   }

   public List<String> getRolesAssigned()
   {
      return auth.getRolesAssigned();
   }

   public String getUsername()
   {
      return auth.getUsername();
   }

   public void logout()
   {
      auth.doLogout();
   }
}
