/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.http.client.*;
import com.gwtext.client.data.*;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.MessageBoxConfig;
import com.gwtext.client.widgets.grid.ColumnConfig;
import com.gwtext.client.widgets.grid.ColumnModel;
import com.gwtext.client.widgets.grid.RowSelectionModel;
import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.util.ModelModificationCallback;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.widgets.RemoteListView;

import java.util.*;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class ProcessDefinitionList extends RemoteListView
{
   private Map<Integer, ProcessDefinitionRef> row2ProcessMap = new HashMap<Integer, ProcessDefinitionRef>();

   public ProcessDefinitionList(ModelModificationCallback callback, String titleName, final MainView view)
   {
      super(callback, titleName, view, getResourceUrl(view));
      enableAddBtn(false);
      final RowSelectionModel sm = new RowSelectionModel(true);
      setRowSelectionModel(sm);      
   }

   private static String getResourceUrl(MainView view)
   {
      return view.getUrlBuilder().getProcessDefinitionsURL();
   }

   protected ColumnModel createColumnModel()
   {
      final ColumnModel columnModel = new ColumnModel(
        new ColumnConfig[]
          {
            new ColumnConfig("Process ID", "processId", 75, true),
            new ColumnConfig("Name", "name", 300, true, null, "expand"),
            new ColumnConfig("Version", "version", 75, true)
          }
      );
      return columnModel;
   }

   protected JsonReader createReader()
   {
      final RecordDef recordDef = new RecordDef(
        new FieldDef[]{
          new IntegerFieldDef("processId"),
          new StringFieldDef("name"),
          new StringFieldDef("version")
        }
      );

      final JsonReader reader = new JsonReader(recordDef);
      reader.setRoot("definitions");
      reader.setTotalProperty("totalCount");
      reader.setId("processId");
      return reader;
   }

   public void onExamine(int row)
   {
      ProcessDefinitionRef proc = row2ProcessMap.get(row);
      launchEditor(proc);
   }

   public void launchEditor(ProcessDefinitionRef proc)
   {
      String editorId = ProcessInstanceListEditor.createWidgetID(proc);

      if(view.hasEditor(editorId))
         view.showEditor(editorId);
      else
         view.addEditor( new ProcessInstanceListEditor(proc, view) );
   }

   public void onDelete(final int row)
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("Delete process?");
            setMsg("Deleting the process will remove all instances as well. " +
              "Would you like to delete this process?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     ProcessDefinitionRef proc = (ProcessDefinitionRef)row2ProcessMap.get(row);
                     String url = view.getUrlBuilder().getRemoveDefinitionURL(proc.getProcessId());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                modelModificationCallback.onStaleModel();
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                ConsoleLog.error("Failed to remove process definition", t);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        ConsoleLog.error("Request failed", e1);
                     }
                  }
               }
            });
         }
      });
   }


   public void onAdd()
   {
      
   }

   protected void onRecordsLoaded(Record[] records)
   {
      row2ProcessMap.clear();

      int i = 0;
      for(Record r : records)
      {
         Long id = Long.valueOf(r.getAsString("processId"));
         ProcessDefinitionRef pd = new ProcessDefinitionRef(
           id,
           r.getAsString("name"),
           r.getAsString("version")
         );

         row2ProcessMap.put(i, pd);
         i++;
      }

      ConsoleLog.debug("Loaded " + row2ProcessMap.size() + " process definitions");
   }

   public List<ProcessDefinitionRef> getAvailableProcessDefinitions()
   {
      List<ProcessDefinitionRef> results = new ArrayList<ProcessDefinitionRef>();
      Iterator<Integer> keys = row2ProcessMap.keySet().iterator();
      while(keys.hasNext())
      {
         results.add(row2ProcessMap.get(keys.next()));
      }

      return results;
   }
}

