/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.util.ModelListenerRegistry;
import com.google.gwt.http.client.*;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.widgets.Button;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.form.DateField;
import com.gwtext.client.widgets.form.FormPanel;
import com.gwtext.client.widgets.form.TextField;
import com.gwtext.client.data.Record;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.widgets.FormWidgets;
import org.jboss.bpm.console.client.util.ModelChangeListener;
import org.jboss.bpm.console.client.util.ModelModificationCallback;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class ProcessInstanceDetailForm extends Panel implements ModelChangeListener
{

   private FormPanel detailsForm;

   private ProcessInstanceRef selectedInstance;

   private ModelModificationCallback callback;

   private MainView mainView;

   private Button suspendButton;
   private Button resumeButton;

   public ProcessInstanceDetailForm(MainView view, ModelModificationCallback callback)
   {
      super();

      this.mainView = view;
      this.callback = callback;

      setTitle("Instance Details");
      setBorder(false);
      setFrame(false);

      // ---------------

      detailsForm = FormWidgets.createBaseFormPanel();

      // ---------------

      TextField idField = new TextField("Id", "instanceId", 230);
      idField.setReadOnly(true);
      detailsForm.add(idField);
      TextField keyField = new TextField("Key", "key", 230);
      keyField.setReadOnly(true);
      detailsForm.add(keyField);
      TextField stateField = new TextField("State", "state", 230);
      stateField.setReadOnly(true);
      detailsForm.add(stateField);
      DateField startDateField = new DateField("Start Date", "startDate", 230);
      startDateField.setReadOnly(true);
      detailsForm.add(startDateField);
      //detailsForm.add(new DateField("End Date", "endDate", 230));

      suspendButton = new Button("Suspend",
            new ButtonListenerAdapter()
            {
               public void onClick(Button button, EventObject eventObject)
               {
                  selectedInstance.setState(ProcessInstanceRef.STATE.SUSPENDED);
                  persistStateChange();
               }
            }
      );

      resumeButton = new Button("Resume",
            new ButtonListenerAdapter()
            {

               public void onClick(Button button, EventObject eventObject)
               {
                  selectedInstance.setState(ProcessInstanceRef.STATE.RUNNING);
                  persistStateChange();
               }
            }
      );

      detailsForm.addButton(suspendButton);
      detailsForm.addButton(resumeButton);

      this.add(detailsForm);
            
   }

   private void persistStateChange()
      {
         String url = mainView.getUrlBuilder().getStateChangeURL(
               selectedInstance.getInstanceId(), selectedInstance.getState()
         );

         RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

         try
         {
            rb.sendRequest(null,
                  new RequestCallback() {

                     public void onResponseReceived(Request request, Response response)
                     {
                        if(response.getStatusCode()!=200)
                           ConsoleLog.error("Failed to change state: " +response.getStatusText());
                        else
                           callback.onStaleModel();
                     }

                     public void onError(Request request, Throwable t)
                     {
                        // Unknown error
                        ConsoleLog.error("Unknown error", t);
                     }
                  });
         }
         catch (RequestException e1)
         {
            ConsoleLog.error("Request failed", e1);
         }
      }


   public void onModelChange(ModelListenerRegistry parent, Object changeEvent)
   {
      // not used
   }

   public void onRecordChange(ModelListenerRegistry parent, Record record)
   {
      selectedInstance = ProcessInstanceList.transform(record);

      if(selectedInstance.isSuspended())
      {
         suspendButton.disable();
         resumeButton.enable();
      }
      else
      {
         resumeButton.disable();
         suspendButton.enable();
      }

      detailsForm.getForm().loadRecord(record);

      // state not part of the payload
      detailsForm.getForm().findField("state").setRawValue(
            selectedInstance.getState().toString()
      );

      detailsForm.doLayout();

   }

   public void onReset()
   {
      selectedInstance = null;
      detailsForm.getForm().reset();
   }
}
