/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.http.client.*;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.json.client.JSONObject;
import com.gwtext.client.data.*;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.MessageBoxConfig;
import com.gwtext.client.widgets.grid.ColumnConfig;
import com.gwtext.client.widgets.grid.ColumnModel;
import com.gwtext.client.widgets.grid.Renderer;
import com.gwtext.client.widgets.grid.CellMetadata;
import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.util.DateRenderer;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.model.DTOParser;
import org.jboss.bpm.console.client.widgets.RemoteListView;
import org.jboss.bpm.console.client.util.ModelModificationCallback;

import java.util.HashMap;
import java.util.Map;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class ProcessInstanceList extends RemoteListView
{

   private Map<Integer, ProcessInstanceRef> row2InstanceMap = new HashMap<Integer, ProcessInstanceRef>();
   private ProcessDefinitionRef parent;
   
   public ProcessInstanceList(ModelModificationCallback callback, ProcessDefinitionRef procDef, String titleName, MainView view)
   {
      super(callback,titleName, view, getResourceUrl(view, procDef.getProcessId()));
      this.parent = procDef;
   }

   ProcessDefinitionRef getProcessDefinition()
   {
      return parent;
   }

   public void onExamine(final int row)
   {
      ProcessInstanceRef instance = row2InstanceMap.get(row);
      String id = ProcessInstanceEditor.ID+"."+instance.getInstanceId();
      if(view.hasEditor(id))
         view.showEditor(id);
      else
         view.addEditor( new ProcessInstanceEditor(parent, instance, view) );
   }

   public void onDelete(final int row)
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("Delete process instance?");
            setMsg("Would you like to delete this process instance?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     ProcessInstanceRef instance = row2InstanceMap.get(row);
                     instance.setState(ProcessInstanceRef.STATE.ENDED);
                     String url = view.getUrlBuilder().getStateChangeURL(instance.getInstanceId(), instance.getState());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                modelModificationCallback.onStaleModel();
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                ConsoleLog.error("Failed to remove process instance", t);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        ConsoleLog.error("Request failed", e1);
                     }
                  }
               }
            });
         }
      });
   }


   public void onAdd()
   {
      MessageBox.show(new MessageBoxConfig() {
         {
            setTitle("New instance");
            setMsg("Would you like to start a new instance?");
            setButtons(MessageBox.YESNOCANCEL);
            setCallback(new MessageBox.PromptCallback() {
               public void execute(String btnID, String text)
               {
                  if("yes".equals( btnID) )
                  {
                     String url = view.getUrlBuilder().getStartNewInstanceURL(parent.getProcessId());
                     RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

                     try
                     {
                        rb.sendRequest("",
                          new RequestCallback() {

                             public void onResponseReceived(Request request, Response response)
                             {
                                if(response.getStatusCode()!=200)
                                {
                                   ConsoleLog.error("Failed to start new instance: " +response.getStatusText());
                                }
                                else
                                {
                                 modelModificationCallback.onStaleModel();
                                 doLayout();
                                }
                             }

                             public void onError(Request request, Throwable t)
                             {
                                // auth failed
                                ConsoleLog.error("Failed to start new instance", t);
                             }
                          });
                     }
                     catch (RequestException e1)
                     {
                        ConsoleLog.error("Request failed", e1);
                     }
                  }
               }
            });
         }
      });
   }

   protected void onRecordsLoaded(Record[] records)
   {
      int i = 0;
      for(Record r : records)
      {
         ProcessInstanceRef pd = transform(r);
         row2InstanceMap.put(i, pd);
         i++;
      }

      ConsoleLog.debug("Loaded " + row2InstanceMap.size() + " process instances");
   }

   public static ProcessInstanceRef transform(Record r)
   {
      ProcessInstanceRef processInstance = null;
      try
      {
         JavaScriptObject js = r.getDataAsJsObject();
         JSONObject jso = new JSONObject(js);
         processInstance = DTOParser.parseProcessInstance(jso);
      }
      catch (Throwable t)
      {
         ConsoleLog.error("Failed to parse process instance", t);
      } 
      
      return processInstance;
   }

   protected ColumnModel createColumnModel()
   {      
      final ColumnModel columnModel = new ColumnModel(
        new ColumnConfig[]
          {
            new ColumnConfig("Instance ID", "instanceId", 75, true),
            //new ColumnConfig("Process ID", "parentId", 25, true),
            //new ColumnConfig("Key", "key", 50, true),
            new ColumnConfig("State", "state", 100, true,
                  new Renderer()
                  {
                     public String render(Object object, CellMetadata cellMetadata, Record record, int i, int i1, Store store)
                     {
                        ProcessInstanceRef tmp = transform(record);
                        return tmp.getState().toString();
                     }
                  }
                  , "expand"),
            new ColumnConfig("Start Date", "startDate", 125, true, new DateRenderer("startDate")),
            //new ColumnConfig("End Date", "endDate", 125, true, new DateRenderer("endDate"))
          }
      );

      return columnModel;
   }

   protected JsonReader createReader()
   {
      final RecordDef recordDef = new RecordDef(
        new FieldDef[]{
          new IntegerFieldDef("instanceId"),
          new IntegerFieldDef("parentId"),
          new StringFieldDef("key"),
          new StringFieldDef("state"),
          new DateFieldDef("startDate", UIConstants.DATE_FORMAT),
          new DateFieldDef("endDate", UIConstants.DATE_FORMAT),
          new BooleanFieldDef("suspended")
        }
      );

      final JsonReader reader = new JsonReader(recordDef);
      reader.setRoot("instances");
      reader.setTotalProperty("totalCount");
      reader.setId("instanceId");

      return reader;
   }

   private static String getResourceUrl(MainView view, long parentId)
   {
      return view.getUrlBuilder().getProcessInstancesURL(parentId);
   }
}
